/**
 * ActionMove.java
 * v1.0
 * 
 * by Matija Tomaskovic
 * (mataya@unforgettable.com || http://www.foi.hr/~mtomasko)
 * 
 * Use this object to move your object from start to end position,
 * in specified period.
 * 
 *** Usage ***
 * 
 * 1. Create this object:
 * 
 *		ActionMove a = new ActionMove(0, 0, 100, 100, 1000);	// initial start & end position, with period
 * 
 * 2. Change start or end target or period whenever you want:
 * 
 *		a.setPeriod(500);	// in milliseconds
 *		a.setStartPosition(100, 100);
 *		a.setEndPosition(100, 100);
 * 
 * 3. Use this to put object to initial position:
 * 
 *		a.doAction(yourObject, 0);
 * 
 * 4. When you have some time to process:
 * 
 *		a.doAction(yourObject, lTimeForAction);
 * 
 * 5. You can check if object has finished moving:
 * 
 *		if (a.m_bFinished)
 *			...
 * 
 *** LOG ***
 * 31.Dec.1999.
 * ~~~~~~~~~~~~
 *	14:23..	- created and v1.0 finished
 */

public class ActionMove
{
	
	private int m_iStartX;
	private int m_iStartY;
	private int m_iEndX;
	private int m_iEndY;
	
	private long m_lPeriod;
	private long m_lTotalTime;		// cumulative of given time
	
	public boolean m_bFinished;
	
	/** CONSTRUCTION *******************************************************/
	
	public ActionMove()
	{
		m_iStartX = 0;
		m_iStartY = 0;
		m_iEndX = 0;
		m_iEndY = 0;
		m_lPeriod = 1000;
		m_lTotalTime = 0;
		
		m_bFinished = true;
	}

	public ActionMove(int iStartX, int iStartY, int iEndX, int iEndY, long lPeriod)
	{
		m_iStartX = iStartX;
		m_iStartY = iStartY;
		m_iEndX = iEndX;
		m_iEndY = iEndY;
		m_lPeriod = lPeriod;
		m_lTotalTime = 0;
		
		m_bFinished = false;
	}

	/** INITIALIZATION *****************************************************/
	
	public void setStartPosition(int x, int y)
	{
		m_iStartX = x;
		m_iStartY = y;
		m_bFinished = false;
	}
	
	public void setEndPosition(int x, int y)
	{
		m_iEndX = x;
		m_iEndY = y;
		m_bFinished = false;
	}

	public void setPeriod(long lPeriod)
	{
		m_lPeriod = lPeriod;
		m_bFinished = false;
	}
	
	/** ACTION HANDLER *****************************************************/

	/**
	 * Spends given time on action, and updates m_bFlag & m_bChanged
	 */
	public void doAction(BackBuffer bbObject, long lTimeForAction)
	{
		if (m_lTotalTime >= m_lPeriod)
		{
			m_bFinished = true;
			return;
		}
		
		m_lTotalTime += lTimeForAction;
		if (m_lTotalTime > m_lPeriod)
			m_lTotalTime = m_lPeriod;

		// calculate new position

		int iCurrentX;
		iCurrentX = (int) (m_iStartX + (m_iEndX - m_iStartX) * m_lTotalTime / m_lPeriod);
		int iCurrentY;
		iCurrentY = (int) (m_iStartY + (m_iEndY - m_iStartY) * m_lTotalTime / m_lPeriod);
		
		bbObject.moveObject(iCurrentX, iCurrentY);
	}

	/**
	 * Resets move action..
	 */
	public void reset()
	{
		m_lTotalTime = 0;
		m_bFinished = false;
	}
}

