/**
 *
 * BadGuy.java
 * v1.0
 * 
 * Project:	Coral Game
 *
 * by Matija Tomaskovic
 * (mataya@unforgettable.com || http://www.foi.hr/~mtomasko)
 *
 *** DESCRIPTION & USAGE ***
 *		
 * This class represents Bad Guy. This one is usually used in some
 * shoot area.
 * 
 * To use it, do following:
 * 
 * 1. Create it, and initialize it:
 * 
 * 		BadGuy	guy = new BadGuy(m_imgWndBadGuyStanding, m_imgWndBadGuyDead, 4, 4, 4, m_component, m_imob);
 * 		guy.setDyingFrame(0, m_imgWndGuyDying1, 0, 0, 100);
 * 		guy.setDyingFrame(1, m_imgWndGuyDying2, 0, 0, 100);
 * 		guy.setDyingFrame(2, m_imgWndGuyDying3, 0, 0, 100);
 * 		guy.setDyingFrame(3, m_imgWndGuyDying4, 0, 0, 100);
 * 		guy.setTriggeringFrame(0, m_imgWndBadGuyTriggering1, 0, 0, 400);
 * 		guy.setTriggeringFrame(1, m_imgWndBadGuyTriggering2, 0, 0, 400);
 * 		guy.setTriggeringFrame(2, m_imgWndBadGuyTriggering3, 0, 0, 400);
 * 		guy.setTriggeringFrame(3, m_imgWndBadGuyTriggering4, 0, 0, 400);
 * 		guy.setShootingFrame(0, m_imgWndBadGuyShooting1, 0, 0, 100);
 * 		guy.setShootingFrame(1, m_imgWndBadGuyShooting2, 0, 0, 100);
 * 		guy.setShootingFrame(2, m_imgWndBadGuyShooting3, 0, 0, 100);
 * 		guy.setShootingFrame(3, m_imgWndBadGuyShooting4, 0, 0, 100);
 * 		guy.m_lMinStandingDelay = 500;
 * 		guy.m_lMinStandingDelay = 2000;
 * 		
 * 		guy.moveObject(100, 200);
 * 
 * 2. Optionally, give him hat:
 * 
 * 		guy.setHat(m_imgHat, 90, 180, m_imgHatDown, m_imgHatDownScore);
 * 
 * 3. Activate it - guy will add himself (and his hat) to parent:
 * 
 *		guy.activate(parent);
 * 
 * 4. Here, guy is doing his job... because parent timing/action
 *	  mechanism is working in the background.
 *	  
 *    First, guy will be in STATE_STANDING, waiting to begin triggering.
 *	  Then, guy begins STATE_TRIGGERING. 
 *    In STATE_STANDING and STATE_TRIGGERING you can shoot guy or his hat
 *    with calling:
 *		
 *		guy.onMouseClick(x, y);
 * 
 *	  If hat is shot, there is RESPONSE_HAT_IS_SHOT, or if guy is shot,
 *	  guy goes to STATE_DYING with RESPONSE_GUY_IS_SHOT (this will end with 
 *    STATE_DEAD and RESPONSE_GUY_IS_DEAD).
 *	  If not shoot, after STATE_TRIGGERING, guy will go to STATE_SHOOT 
 *    and set RESPONSE_GUY_SHOOT.
 * 
 *  5. At the end call: 
 * 
 *		guy.dismiss(parent);	// guy will remove itself and his hat from parent
 *
 *
 *** LOG ***
 * 
 * 23.Dec.1999.
 * ~~~~~~~~~~~~
 *	15:18..	- created
 *
 * 31.Dec.1999.
 * ~~~~~~~~~~~~
 *	11:39..	- v1.0 is finished!
 *
 */
import java.awt.*;
import java.awt.image.ImageObserver;
import java.awt.image.PixelGrabber;
import java.lang.Math;


public class BadGuy
	extends ActiveBackBuffer
{
	/** IMAGES *************************************************************/

	Image	m_imgGuyStanding;
	Image	m_imgGuyDead;
	Image	m_imgGuyDeadToEarly;
	
	ImageFrames	m_GuyDyingFrames;
	

	/** STATES *************************************************************/
	
	final static int STATE_STANDING			= 1;	// guy is standing & waiting for triggering
	final static int STATE_DYING			= 2;	// guy was just shoot and is dying...
	final static int STATE_DEAD				= 3;	// guy was shoot and is lying dead
	final static int STATE_TRIGGERING		= 4;	// guy is pulling the trigger
	final static int STATE_SHOOT			= 5;	// guy is shooting
	final static int STATE_DONE_SHOOTING	= 6;	// guy is finished shooting, and is waiting for next command
	
	int m_iState;

	
	final static int RESPONSE_NONE				= 0;
	final static int RESPONSE_GUY_IS_SHOT		= 1;	// guy is being shot from player
	final static int RESPONSE_GUY_IS_SHOT_TOO_EARLY	= 2;	// guy is being shot from player but before pulling trigger
	final static int RESPONSE_GUY_IS_DEAD		= 3;	// guy's hat is shot from player
	final static int RESPONSE_HAT_IS_SHOT		= 4;
	final static int RESPONSE_GUY_SHOOT			= 5;	// guy did shoot
	
	int m_iResponse = RESPONSE_NONE;

	/** IMAGES *************************************************************/

	ImageFrames		m_GuyTriggeringFrames;
	ImageFrames		m_GuyShootingFrames;

	Hat	m_hat;
	
	/** CONSTRUCTION *******************************************************/

	public BadGuy(Image imgGuyStanding, Image imgGuyDead, Image imgGuyDeadToEarly,
				  int iMaxDyingFrames, int iMaxTriggeringFrames, int iMaxShootingFrames,
				  Component c, ImageObserver imob)
	{
		createForImage(c, imob);
		m_strName = "BADGUY";
		
		// remember given images
		m_imgGuyStanding = imgGuyStanding;
		m_imgGuyDead = imgGuyDead;
		m_imgGuyDeadToEarly = imgGuyDeadToEarly;
		
		m_GuyDyingFrames = new ImageFrames(iMaxDyingFrames);
		m_GuyTriggeringFrames = new ImageFrames(iMaxTriggeringFrames);
		m_GuyShootingFrames = new ImageFrames(iMaxShootingFrames);
		
		setStateStanding();
	}

	/** INITIALIZATION *****************************************************/
	
	public void setTriggeringFrame(int iFrame, Image img, int iX, int iY, long lDelay)
	{
		m_GuyTriggeringFrames.setFrame(iFrame, img, iX, iY, lDelay);
	}

	public void setShootingFrame(int iFrame, Image img, int iX, int iY, long lDelay)
	{
		m_GuyShootingFrames.setFrame(iFrame, img, iX, iY, lDelay);
	}
	
	public void setDyingFrame(int iFrame, Image imgFrame, int x, int y, long lDelay)
	{
		m_GuyDyingFrames.setFrame(iFrame, imgFrame, x, y, lDelay);
	}
	
	public void setHat(Image imgHat, int x, int y, Image imgHatDown, Image imgHatDownScore)
	{
		m_hat = new Hat(imgHat, x, y, 
						imgHatDown, this.y + m_imgGuyStanding.getHeight(m_imob) - imgHatDown.getHeight(m_imob), imgHatDownScore,
						m_component, m_imob);
	}
	
	
	/** ACTION *************************************************************/
	
	long m_lMinStandingDelay = 2000;	// min. delay (in milliseconds) to wait to start Standing
	long m_lMaxStandingDelay = 4000;	// max. delay (in milliseconds) to wait to start Standing

	public void setStateStanding()
	{
		m_iState = STATE_STANDING;
		m_iResponse = RESPONSE_NONE;
		
		setImage(m_imgGuyStanding);
		
		// calculate delay until triggering...
	
		double d = m_lMinStandingDelay + (Math.random() * (m_lMaxStandingDelay - m_lMinStandingDelay));
		Double dd = new Double(d);
		setActionWhenTime(dd.longValue());
	}
	
	boolean m_bShotToEarly = false;
	public void setStateDying()
	{
		if (m_iState == STATE_STANDING)
		{
			m_iResponse = RESPONSE_GUY_IS_SHOT_TOO_EARLY;
			m_bShotToEarly = true;
		}
		else
		{
			m_iResponse = RESPONSE_GUY_IS_SHOT;
			m_bShotToEarly = false;
		}
			
		m_iState = STATE_DYING;		
		
		// if there is hat..
		if (m_hat != null)
		{
			if (m_hat.m_iState == m_hat.STATE_ON_HEAD)
				m_hat.setStateShooted();
		}

		// set first frame of dying
		m_GuyDyingFrames.m_iCurrentFrame = 0;
		Image img = m_GuyDyingFrames.getCurrentImage();
		if (img != null)
		{
			setImage(img);
			setActionWhenTime(m_GuyDyingFrames.getCurrentDelay());
		}
		// or if no dying frames - go to STATE_DEAD directly
		else
		{
			setStateDead();
		}
	}
	
	public void setStateDead()
	{
		m_iState = STATE_DEAD;
		m_iResponse = RESPONSE_GUY_IS_DEAD;
		
		if (m_bShotToEarly)
			setImage(m_imgGuyDeadToEarly);
		else
			setImage(m_imgGuyDead);
		setActionNone();
	}
	

	public void setStateShoot()
	{
		m_iState = STATE_SHOOT;
		m_iResponse = RESPONSE_GUY_SHOOT;
				
		m_GuyShootingFrames.m_iCurrentFrame = 0;
		Image img = m_GuyShootingFrames.getCurrentImage();
		if (img != null)
		{
			setImage(img);
			setActionWhenTime(m_GuyShootingFrames.getCurrentDelay());
		}
	}
	
	public void setStateDoneShooting()
	{
		m_iState = STATE_DONE_SHOOTING;
		setActionNone();				
	}
	
	
	/**
	 * This will do next action, and set
	 */
	public void doNextAction()
	{
		if (m_iState == STATE_STANDING)
			doStateStanding();
		else if (m_iState == STATE_TRIGGERING)
			doStateTriggering();
		else if (m_iState == STATE_SHOOT)
			doStateShooting();
		else if (m_iState == STATE_DYING)
			doStateDying();
		else if (m_iState == STATE_DEAD)
			doStateDead();
		else if (m_iState == STATE_DONE_SHOOTING)
			doStateDoneShooting();
	}
	
	private void doStateStanding()
	{
		// time has passed (set in setStateStanding() - begin triggering
		m_iState = STATE_TRIGGERING;

		m_GuyTriggeringFrames.m_iCurrentFrame = 0;
		Image img = m_GuyTriggeringFrames.getCurrentImage();
		if (img != null)
		{
			setImage(img);
			setActionWhenTime(m_GuyTriggeringFrames.getCurrentDelay());
		}
	}
	
	private void doStateTriggering()
	{
		// do next frame..
		if (m_GuyTriggeringFrames.nextFrame())
		{
			setImage(m_GuyTriggeringFrames.getCurrentImage());
			setActionWhenTime(m_GuyTriggeringFrames.getCurrentDelay());
		}
		// if no more triggering frames..
		else
		{
			setStateShoot();
		}
	}

	private void doStateShooting()
	{
		// do next frame..
		if (m_GuyShootingFrames.nextFrame())
		{
			setImage(m_GuyShootingFrames.getCurrentImage());
			setActionWhenTime(m_GuyShootingFrames.getCurrentDelay());
		}
		// if no more triggering frames..
		else
		{
			setStateDoneShooting();
		}
	}
	
	private void doStateDying()
	{
		// do next frame..
		if (m_GuyDyingFrames.nextFrame())
		{
			setImage(m_GuyDyingFrames.getCurrentImage());
			setActionWhenTime(m_GuyDyingFrames.getCurrentDelay());
		}
		// if no more dying frames..
		else
		{
			setStateDead();
		}
	}
	
	private void doStateDead()
	{
		setActionNone();
	}

	private void doStateDoneShooting()
	{
		setActionNone();
	}


	/** UTILITY ************************************************************/
	
	public void activate(ActiveBackBuffer parent)
	{
		parent.addObject(this);
		parent.moveObjectToBack(this);
		
		// check to add hat
		if (m_hat != null)
		{
			m_hat.setStateOnHead();
			parent.removeObject(m_hat);
			parent.addObject(m_hat);
			m_hat.moveObject(m_hat.m_iHatX, m_hat.m_iHatY);
			parent.moveObjectToBack(m_hat);
		}
		
		setStateStanding();
	}

	
	public void dismiss(ActiveBackBuffer parent)
	{
		if (m_hat != null)
		{
			parent.removeObject(m_hat);
		}
					
		parent.removeObject(this);
	}
	
	public void onMouseClick(int x, int y)
	{
		if ((m_iState == STATE_STANDING) ||
			(m_iState == STATE_TRIGGERING))
		{
			if (hasPixelAt(x, y))
			{
				setStateDying();
			}
			else
			{
				if (m_hat != null)
				{
					if (m_hat.m_iState == m_hat.STATE_ON_HEAD)
					{
						if (m_hat.hasPixelAt(x, y))
						{
							m_hat.setStateShootedWithScore();
							m_iResponse = RESPONSE_HAT_IS_SHOT;
						}
					}
				}
			}
		}
	}
}
