/**
 *
 * CoralGame.java - main engine class for Coral Game
 *
 * by Matija Tomaskovic
 * (mataya@unforgettable.com || http://www.foi.hr/~mtomasko)
 *
 *** DESCRIPTION AND USAGE ***
 * 
 * CoralGame represents main game engine class.
 * 
 * Applet will create one instance of this class, initialize it, and
 * then call:
 *	- mouse handlers
 *	- keyboard handlers
 *	- graphics handlers (paint & update)
 *	- action handlers
 * while this engine will run game.
 * 
 * Steps:
 * 
 * 1. Create instance of this class:
 * 
 *		CoralGame m_game = new CoralGame();
 * 
 * 2. Load all images (start loading) of this game:
 * 
 * 		for ( int i=0; i < m_game.NUMBER_OF_IMAGES; i++ )
 * 		{
 * 			m_game.img[i] = getImage(getDocumentBase(), "gfx/" + m_game.imgName[i]);
 * 			// optionally, you may use tracker:
 *			tracker.addImage(m_game.img[i], iTrackerImgCount++);
 * 		}
 * 
 * 3. After you've make sure all images are loaded (using tracker), you must
 *    run initialization of the game:
 * 
 *		m_game.init(yourComponent, yourImageObserver);
 * 
 *	  This will:
 *		- initialize game variables and objects
 *		- prepare all for 1st level of HotelGame
 *		- launch STATE_GET_READY
 * 
 * 4. From this point on, you can:
 * 
 * - Use mouse handling:
 * 
 *		m_game.onMouseMoved(e.getX(), e.getY());
 * 		m_game.onMouseClicked(e.getX(), e.getY());
 * 
 * 		m_iMouseCursor = m_game.m_iMouseCursor;
 * 		if (m_iMouseCursor == m_game.MOUSE_CURSOR_ARROW)
 * 			setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
 * 		else if (m_iMouseCursor == m_game.MOUSE_CURSOR_HAND)
 * 			setCursor(new Cursor(Cursor.HAND_CURSOR));
 * 
 * - Use keyboard handling:
 * 
 *		m_game.onKeyEsc();
 * 
 * - Use graphics handling:
 * 
 *		m_game.paint(g, 0, 0);
 *		:
 *		m_game.update(g, 0, 0);
 * 
 * - Use ACTION handler (this is what makes game running !!!!!!)
 * 
 *		m_game.action(lTimeSpendOnAction);
 * 
 * 
 * E.g. if you are using thread in applet, your run() handler could look like this:
 * 
 *	public void run()
 *	{
 *		loadImagesAndSounds();
 *		
 *		m_game.init();
 *		repaint();
 *
 *		//
 *		// Get "first time"
 *		//
 *		long lNewTime;
 *		long lLastTime = System.currentTimeMillis();
 *		
 *		//
 *		// game-loop
 *		//
 *		while(true)
 *		{
 *			lNewTime = System.currentTimeMillis();
 *
 *			// do action
 *			m_game.action(lNewTime - lLastTime);
 *			
 *			// repaint
 *			update(getGraphics());
 *
 *			updateMouseCursor();
 *			doSoundEffects();
 *			
 *			lLastTime = lNewTime;
 *			
 *			// give little time to system...
 *			try
 *			{
 *				m_threadGame.sleep(10);
 *			}
 *			catch(InterruptedException e) {}
 *		}
 *	}
 *
 *** LOG ***
 * 
 * 08.Aug.1999.
 * ~~~~~~~~~~~~
 *	- created
 * 
 * 21.Dec.1999.
 * ~~~~~~~~~~~~
 *	22:32..	- redesigning...
 *	..00:39	- STATE_* added.. 
 *			- finished: STATE_INITIALIZING, STATE_GET_READY, 
 *			  STATE_CHECK_TO_EXIT_GAME,	STATE_GAME_OVER, and shell for STATE_PLAYING
 *	..01:14 - finished painting of lives
 *	..01:24	- finished painting bullets
 *	..01:46 - finished painting scores
 *			- "Get Ready!" is now blinking..
 * 
 * 26.Dec.1999.
 * ~~~~~~~~~~~~
 *	..18:31	- splitted into CoralGame with HotelGame and FightGame
 * 
 * 03.Jan.1999.
 * ~~~~~~~~~~~~
 *	14:03 - added small cheat for levels..
 */

import java.awt.*;
import java.awt.image.*;
import java.awt.image.ImageObserver;

public class CoralGame
{
	/** PARENT VARIABLES ***************************************************/
	
	ImageObserver	m_imob;
	Component		m_component;

	/** IMAGES *****************************************************************
	 * Rules:
	 * 1. Every image has its unique index
	 * 2. With image index - we can get image filename in imgName[] array
	 * 3. With image index - we can get image object in img[] array
	 */
	final static int IMG_BACKGROUND_FIGHT		= 0;
	final static int IMG_BACKGROUND				= 1;
	final static int IMG_DOOR_CLOSED			= 2;
	final static int IMG_DOOR_OPENING1			= 3;
	final static int IMG_DOOR_OPENING2			= 4;
	final static int IMG_DOOR_OPENING3			= 5;
	final static int IMG_DOOR_OPENING4			= 6;
	final static int IMG_DOOR_OPENED			= 7;
	final static int IMG_WINDOW_CLOSED			= 8;
	final static int IMG_WINDOW_OPENING1		= 9;
	final static int IMG_WINDOW_OPENING2		= 10;
	final static int IMG_WINDOW_OPENING3		= 11;
	final static int IMG_WINDOW_OPENED			= 12;
	final static int IMG_MOUSE					= 13;
	final static int IMG_BADGUY_STANDING		= 14;
	final static int IMG_BADGUY_TRIGGERING1		= 15;
	final static int IMG_BADGUY_TRIGGERING2		= 16;
	final static int IMG_BADGUY_TRIGGERING3		= 17;
	final static int IMG_BADGUY_TRIGGERING4		= 18;
	final static int IMG_BADGUY_SHOOTING1		= 19;
	final static int IMG_BADGUY_SHOOTING2		= 20;
	final static int IMG_BADGUY_SHOOTING3		= 21;
	final static int IMG_BADGUY_SHOOTING4		= 22;
	final static int IMG_BADGUY_DYING1			= 23;
	final static int IMG_BADGUY_DYING2			= 24;
	final static int IMG_BADGUY_DYING3			= 25;
	final static int IMG_BADGUY_DYING4			= 26;
	final static int IMG_BAD_GUY_DEAD			= 27;
	final static int IMG_GOOD_GUY_DEAD			= 28;
	
	final static int IMG_WND_GOOD_GUY_STANDING		= 29;
	final static int IMG_WND_BAD_GUY_DEAD			= 30;
	final static int IMG_WND_GOOD_GUY_DEAD			= 31;
	final static int IMG_WND_GUY_DYING1				= 32;
	final static int IMG_WND_GUY_DYING2				= 33;
	final static int IMG_WND_GUY_DYING3				= 34;
	final static int IMG_WND_GUY_DYING4				= 35;
	final static int IMG_WND_GUY_DYING5				= 36;
	final static int IMG_WND_BAD_GUY_STANDING		= 37;
	final static int IMG_WND_BAD_GUY_TRIGGERING1	= 38;
	final static int IMG_WND_BAD_GUY_TRIGGERING2	= 39;
	final static int IMG_WND_BAD_GUY_TRIGGERING3	= 40;
	final static int IMG_WND_BAD_GUY_TRIGGERING4	= 41;
	final static int IMG_WND_BAD_GUY_SHOOTING1		= 42;
	final static int IMG_WND_BAD_GUY_SHOOTING2		= 43;
	final static int IMG_WND_BAD_GUY_SHOOTING3		= 44;
	final static int IMG_WND_BAD_GUY_SHOOTING4		= 45;
	final static int IMG_WND_GOOD_BAD_GUY_STANDING	= 46;
	final static int IMG_HAT						= 47;
	final static int IMG_HAT_DOWN					= 48;
	final static int IMG_HAT_DOWN_SCORE				= 49;
	final static int IMG_GOOD_GUY_STANDING			= 50;
	final static int IMG_GOOD_BAD_GUY_STANDING		= 51;
	final static int IMG_GET_READY					= 52;
	final static int IMG_GAME_OVER					= 53;
	final static int IMG_LIFE						= 54;	
	final static int IMG_NUMBER_NULL				= 55;
	final static int IMG_NUMBER_0					= 56;
	final static int IMG_NUMBER_1					= 57;
	final static int IMG_NUMBER_2					= 58;
	final static int IMG_NUMBER_3					= 59;
	final static int IMG_NUMBER_4					= 60;
	final static int IMG_NUMBER_5					= 61;
	final static int IMG_NUMBER_6					= 62;
	final static int IMG_NUMBER_7					= 63;
	final static int IMG_NUMBER_8					= 64;
	final static int IMG_NUMBER_9					= 65;
	final static int IMG_TXT_EXIT_GAME   			= 66;
	final static int IMG_TXT_PREPARE_FOR_BONUS_GAME	= 67;
	final static int IMG_TXT_SORRY                  = 68;
	final static int IMG_TXT_INTRO					= 69;
	final static int IMG_BIG_BADGUY_STANDING		= 70;
	final static int IMG_BIG_BADGUY_TRIGGERING1		= 71;
	final static int IMG_BIG_BADGUY_TRIGGERING2		= 72;
	final static int IMG_BIG_BADGUY_TRIGGERING3		= 73;
	final static int IMG_BIG_BADGUY_TRIGGERING4		= 74;
	final static int IMG_BIG_BADGUY_SHOOTING1		= 75;
	final static int IMG_BIG_BADGUY_SHOOTING2		= 76;
	final static int IMG_BIG_BADGUY_SHOOTING3		= 77;
	final static int IMG_BIG_BADGUY_SHOOTING4		= 78;
	final static int IMG_BIG_BADGUY_DYING1			= 79;
	final static int IMG_BIG_BADGUY_DYING2			= 80;
	final static int IMG_BIG_BADGUY_DYING3			= 81;
	final static int IMG_BIG_BADGUY_DYING4			= 82;
	final static int IMG_BIG_BADGUY_DYING5			= 83;
	final static int IMG_BIG_BAD_GUY_DEAD			= 84;
	final static int IMG_BIG_BAD_GUY_DEAD_TO_EARLY	= 85;
	final static int IMG_BIG_HAT					= 86;
	final static int IMG_BIG_HAT_DOWN				= 87;
	//									---------------
	final static int NUMBER_OF_IMAGES				= 88;  // last index + 1

	// image files to be loaded into img array
	final static String imgName[] = {
		"back_fight.jpg",
		"back.jpg",
		"door_closed.gif",
		"door1.gif",
		"door2.gif",
		"door3.gif",
		"door4.gif",
		"door_opened.gif",
		"window_closed.gif",
		"window1.gif",
		"window2.gif",
		"window3.gif",
		"window_opened.gif",
		"mouse_target.gif",
		"bad_guy_standing.gif",
		"bad_guy_trigger1.gif",
		"bad_guy_trigger2.gif",
		"bad_guy_trigger3.gif",
		"bad_guy_trigger4.gif",
		"bad_guy_shoot1.gif",
		"bad_guy_shoot2.gif",
		"bad_guy_shoot3.gif",
		"bad_guy_shoot4.gif",
		"guy_dying1.gif",
		"guy_dying2.gif",
		"guy_dying3.gif",
		"guy_dying4.gif",
		"bad_guy_dead.gif",
		"good_guy_dead.gif",
		"wnd_good_guy_standing.gif",
		"wnd_bad_guy_dead.gif",
		"wnd_good_guy_dead.gif",
		"wnd_guy_dying1.gif",
		"wnd_guy_dying2.gif",
		"wnd_guy_dying3.gif",
		"wnd_guy_dying4.gif",
		"wnd_guy_dying5.gif",
		"wnd_bad_guy_standing.gif",
		"wnd_bad_guy_trigger1.gif",
		"wnd_bad_guy_trigger2.gif",
		"wnd_bad_guy_trigger3.gif",
		"wnd_bad_guy_trigger4.gif",
		"wnd_bad_guy_shoot1.gif",
		"wnd_bad_guy_shoot2.gif",
		"wnd_bad_guy_shoot3.gif",
		"wnd_bad_guy_shoot4.gif",
		"wnd_good_bad_guy_standing.gif",
		"hat.gif",
		"hat_down.gif",
		"hat_down_score.gif",
		"good_guy_standing.gif",
		"good_bad_guy_standing.gif",
		"text_get_ready.gif",
		"text_game_over.gif",
		"life.gif",
		"num_blanc.gif",
		"num_0.gif",
		"num_1.gif",
		"num_2.gif",
		"num_3.gif",
		"num_4.gif",
		"num_5.gif",
		"num_6.gif",
		"num_7.gif",
		"num_8.gif",
		"num_9.gif",
		"txt_exit_game.gif",
		"txt_prepare_for_bonus_game.gif",
		"txt_sorry.gif",
		"intro.gif",
		"big_bad_guy_standing.gif",
		"big_bad_guy_trigger1.gif",
		"big_bad_guy_trigger2.gif",
		"big_bad_guy_trigger3.gif",
		"big_bad_guy_trigger4.gif",
		"big_bad_guy_shoot1.gif",
		"big_bad_guy_shoot2.gif",
		"big_bad_guy_shoot3.gif",
		"big_bad_guy_shoot4.gif",
		"big_guy_dying1.gif",
		"big_guy_dying2.gif",
		"big_guy_dying3.gif",
		"big_guy_dying4.gif",
		"big_guy_dying5.gif",
		"big_guy_dead.gif",
		"big_guy_dead_to_early.gif",
		"big_hat.gif",
		"big_hat_down.gif"
	};

	// images (loaded by applet), use IMG_* index for access..
	Image m_images[];

	
	
	/** CURRENT GAME *******************************************************/
	
	int	m_iLevel = 1;						// current level - 1, 2,..

	
	/** TEXT OBJECTS *******************************************************/

	boolean __TEXT_OBJECTS________________;	// just for nicer view in Class Outline
	
	final static int TXT_GET_READY_Y				= 170;
	final static int TXT_PREPARE_FOR_BONUS_GAME_Y	= 170;
	final static int TXT_SORRY_Y					= 170;
	final static int TXT_GAME_OVER_Y				= 170;
	final static int TXT_EXIT_GAME_Y				= 170;
	final static int TXT_INTRO_Y					= 40;
	
	final static int TIME_GET_READY_BLINKING		= 300;
	
	BackBuffer	m_bbTxtGetReady;
	BackBuffer	m_bbTxtGameOver;
	BackBuffer	m_bbTxtExitGame;
	BackBuffer	m_bbTxtPrepareForBonusGame;
	BackBuffer	m_bbTxtSorry;
	BackBuffer	m_bbTxtIntro;
	
	ActionBlinking	m_blinker;
	ActionToggle	m_actionMenuItemToggle = new ActionToggle(500);
	
	ActionToggle	m_stateToggler = new ActionToggle(3000);

	
	/**
	 * There are two sub-games in CoralGame - Hotel Game and Fight Game.
	 * HotelGame is one that is first started. Player plays it, until
	 * he gets enough scores to go to next level.
	 * But, to go to next level, player must first pass Fight Game.
	 * If player pass Fight Game, then he continues to go to next level,
	 * else he must play Coral Game for current level again.
	 */
	HotelGame	m_hotel;
	FightGame	m_fight;

	BackBuffer	m_bbMouse;

	/** SOUND EFFECTS FLAGS ************************************************/
	
	boolean m_bSndPlayerGun = false;
	boolean m_bSndPlayerGunHat = false;
	boolean m_bSndFightGun = false;
	boolean m_bSndFightGunHat = false;

	
	/** CODE ***************************************************************/
	
	public CoralGame(Component c, ImageObserver imob)
	{
		m_component = c;
		m_imob = imob;

		// create image array
		m_images = new Image[NUMBER_OF_IMAGES];
		
	}
	

	/**
	 * Initializes game.
	 * This one should be called after all images were loaded.
	 */
	public void init()
	{
		//
		// Initialize hotel
		//
		
		m_hotel = new HotelGame(m_component, m_imob);
		m_hotel.m_imgBackground = m_images[IMG_BACKGROUND];
		m_hotel.m_imgLife = m_images[IMG_LIFE];
	
		m_hotel.m_imgNumberNull = m_images[IMG_NUMBER_NULL];
		m_hotel.m_imgNumber0 = m_images[IMG_NUMBER_0];
		m_hotel.m_imgNumber1 = m_images[IMG_NUMBER_1];
		m_hotel.m_imgNumber2 = m_images[IMG_NUMBER_2];
		m_hotel.m_imgNumber3 = m_images[IMG_NUMBER_3];
		m_hotel.m_imgNumber4 = m_images[IMG_NUMBER_4];
		m_hotel.m_imgNumber5 = m_images[IMG_NUMBER_5];
		m_hotel.m_imgNumber6 = m_images[IMG_NUMBER_6];
		m_hotel.m_imgNumber7 = m_images[IMG_NUMBER_7];
		m_hotel.m_imgNumber8 = m_images[IMG_NUMBER_8];
		m_hotel.m_imgNumber9 = m_images[IMG_NUMBER_9];
	
		m_hotel.m_imgDoorClosed = m_images[IMG_DOOR_CLOSED];
		m_hotel.m_imgDoorOpened = m_images[IMG_DOOR_OPENED];
		m_hotel.m_imgDoorOpening1 = m_images[IMG_DOOR_OPENING1];
		m_hotel.m_imgDoorOpening2 = m_images[IMG_DOOR_OPENING2];
		m_hotel.m_imgDoorOpening3 = m_images[IMG_DOOR_OPENING3];
		m_hotel.m_imgDoorOpening4 = m_images[IMG_DOOR_OPENING4];
	
		m_hotel.m_imgWindowClosed = m_images[IMG_WINDOW_CLOSED];
		m_hotel.m_imgWindowOpened = m_images[IMG_WINDOW_OPENED];
		m_hotel.m_imgWindowOpening1 = m_images[IMG_WINDOW_OPENING1];
		m_hotel.m_imgWindowOpening2 = m_images[IMG_WINDOW_OPENING2];
		m_hotel.m_imgWindowOpening3 = m_images[IMG_WINDOW_OPENING3];
	
		m_hotel.m_imgBadGuyStanding = m_images[IMG_BADGUY_STANDING];
		m_hotel.m_imgBadGuyDead = m_images[IMG_BAD_GUY_DEAD];
		m_hotel.m_imgGoodGuyDead = m_images[IMG_GOOD_GUY_DEAD];
		m_hotel.m_imgBadGuyDying1 = m_images[IMG_BADGUY_DYING1];
		m_hotel.m_imgBadGuyDying2 = m_images[IMG_BADGUY_DYING2];
		m_hotel.m_imgBadGuyDying3 = m_images[IMG_BADGUY_DYING3];
		m_hotel.m_imgBadGuyDying4 = m_images[IMG_BADGUY_DYING4];
		m_hotel.m_imgBadGuyTriggering1 = m_images[IMG_BADGUY_TRIGGERING1];
		m_hotel.m_imgBadGuyTriggering2 = m_images[IMG_BADGUY_TRIGGERING2];
		m_hotel.m_imgBadGuyTriggering3 = m_images[IMG_BADGUY_TRIGGERING3];
		m_hotel.m_imgBadGuyTriggering4 = m_images[IMG_BADGUY_TRIGGERING4];
		m_hotel.m_imgBadGuyShooting1 = m_images[IMG_BADGUY_SHOOTING1];
		m_hotel.m_imgBadGuyShooting2 = m_images[IMG_BADGUY_SHOOTING2];
		m_hotel.m_imgBadGuyShooting3 = m_images[IMG_BADGUY_SHOOTING3];
		m_hotel.m_imgBadGuyShooting4 = m_images[IMG_BADGUY_SHOOTING4];
		
		m_hotel.m_imgWndGoodGuyStanding = m_images[IMG_WND_GOOD_GUY_STANDING];
		m_hotel.m_imgWndBadGuyDead = m_images[IMG_WND_BAD_GUY_DEAD];
		m_hotel.m_imgWndGoodGuyDead = m_images[IMG_WND_GOOD_GUY_DEAD];
		m_hotel.m_imgWndGuyDying1 = m_images[IMG_WND_GUY_DYING1];
		m_hotel.m_imgWndGuyDying2 = m_images[IMG_WND_GUY_DYING2];
		m_hotel.m_imgWndGuyDying3 = m_images[IMG_WND_GUY_DYING3];
		m_hotel.m_imgWndGuyDying4 = m_images[IMG_WND_GUY_DYING4];
		m_hotel.m_imgWndGuyDying5 = m_images[IMG_WND_GUY_DYING5];
		
		m_hotel.m_imgWndBadGuyStanding = m_images[IMG_WND_BAD_GUY_STANDING];
		m_hotel.m_imgWndBadGuyTriggering1 = m_images[IMG_WND_BAD_GUY_TRIGGERING1];
		m_hotel.m_imgWndBadGuyTriggering2 = m_images[IMG_WND_BAD_GUY_TRIGGERING2];
		m_hotel.m_imgWndBadGuyTriggering3 = m_images[IMG_WND_BAD_GUY_TRIGGERING3];
		m_hotel.m_imgWndBadGuyTriggering4 = m_images[IMG_WND_BAD_GUY_TRIGGERING4];
		m_hotel.m_imgWndBadGuyShooting1 = m_images[IMG_WND_BAD_GUY_SHOOTING1];
		m_hotel.m_imgWndBadGuyShooting2 = m_images[IMG_WND_BAD_GUY_SHOOTING2];
		m_hotel.m_imgWndBadGuyShooting3 = m_images[IMG_WND_BAD_GUY_SHOOTING3];
		m_hotel.m_imgWndBadGuyShooting4 = m_images[IMG_WND_BAD_GUY_SHOOTING4];
		
		m_hotel.m_imgWndGoodBadGuyStanding = m_images[IMG_WND_GOOD_BAD_GUY_STANDING];
		
		m_hotel.m_imgGoodGuyStanding = m_images[IMG_GOOD_GUY_STANDING];
		m_hotel.m_imgGoodBadGuyStanding = m_images[IMG_GOOD_BAD_GUY_STANDING];
		
		m_hotel.m_imgHat = m_images[IMG_HAT];
		m_hotel.m_imgHatDown = m_images[IMG_HAT_DOWN];
		m_hotel.m_imgHatDownScore = m_images[IMG_HAT_DOWN_SCORE];

		m_hotel.init();
		
		m_hotel.prepareForNewGame();
		
		//
		// Initialize FightGame
		//
		
		m_fight = new FightGame(m_component, m_imob);
		m_fight.m_imgBackground = m_images[IMG_BACKGROUND_FIGHT];
		m_fight.m_imgLife = m_images[IMG_LIFE];
		m_fight.m_imgBadGuyStanding = m_images[IMG_BIG_BADGUY_STANDING];
		m_fight.m_imgBadGuyDead = m_images[IMG_BIG_BAD_GUY_DEAD];
		m_fight.m_imgBadGuyDeadToEarly = m_images[IMG_BIG_BAD_GUY_DEAD_TO_EARLY];
		m_fight.m_imgBadGuyDying1 = m_images[IMG_BIG_BADGUY_DYING1];
		m_fight.m_imgBadGuyDying2 = m_images[IMG_BIG_BADGUY_DYING2];
		m_fight.m_imgBadGuyDying3 = m_images[IMG_BIG_BADGUY_DYING3];
		m_fight.m_imgBadGuyDying4 = m_images[IMG_BIG_BADGUY_DYING4];
		m_fight.m_imgBadGuyDying5 = m_images[IMG_BIG_BADGUY_DYING5];
		m_fight.m_imgBadGuyTriggering1 = m_images[IMG_BIG_BADGUY_TRIGGERING1];
		m_fight.m_imgBadGuyTriggering2 = m_images[IMG_BIG_BADGUY_TRIGGERING2];
		m_fight.m_imgBadGuyTriggering3 = m_images[IMG_BIG_BADGUY_TRIGGERING3];
		m_fight.m_imgBadGuyTriggering4 = m_images[IMG_BIG_BADGUY_TRIGGERING4];
		m_fight.m_imgBadGuyShooting1 = m_images[IMG_BIG_BADGUY_SHOOTING1];
		m_fight.m_imgBadGuyShooting2 = m_images[IMG_BIG_BADGUY_SHOOTING2];
		m_fight.m_imgBadGuyShooting3 = m_images[IMG_BIG_BADGUY_SHOOTING3];
		m_fight.m_imgBadGuyShooting4 = m_images[IMG_BIG_BADGUY_SHOOTING4];
		m_fight.m_imgHat = m_images[IMG_BIG_HAT];
		m_fight.m_imgHatDown = m_images[IMG_BIG_HAT_DOWN];
		
		m_fight.init();
		
		m_blinker = new ActionBlinking(TIME_GET_READY_BLINKING);
		
		// Prepare "Get Ready!" object
		m_bbTxtGetReady = new BackBuffer();
		m_bbTxtGetReady.createForImage(m_component, m_imob);
		m_bbTxtGetReady.setImage(m_images[IMG_GET_READY]);
		m_bbTxtGetReady.moveObject((640 - m_bbTxtGetReady.width)/2, TXT_GET_READY_Y);
		
		// Prepare "Game Over" object
		m_bbTxtGameOver = new BackBuffer();
		m_bbTxtGameOver.createForImage(m_component, m_imob);
		m_bbTxtGameOver.setImage(m_images[IMG_GAME_OVER]);
		m_bbTxtGameOver.moveObject((640 - m_bbTxtGameOver.width)/2, TXT_GAME_OVER_Y);
		
		m_bbTxtExitGame = new BackBuffer();
		m_bbTxtExitGame.createForImage(m_component, m_imob);
		m_bbTxtExitGame.setImage(m_images[IMG_TXT_EXIT_GAME]);
		m_bbTxtExitGame.moveObject((640 - m_bbTxtExitGame.width)/2, TXT_EXIT_GAME_Y);
		
		m_bbTxtPrepareForBonusGame = new BackBuffer();
		m_bbTxtPrepareForBonusGame.createForImage(m_component, m_imob);
		m_bbTxtPrepareForBonusGame.setImage(m_images[IMG_TXT_PREPARE_FOR_BONUS_GAME]);
		m_bbTxtPrepareForBonusGame.moveObject((640 - m_bbTxtPrepareForBonusGame.width)/2, TXT_PREPARE_FOR_BONUS_GAME_Y);
		
		m_bbTxtSorry = new BackBuffer();
		m_bbTxtSorry.createForImage(m_component, m_imob);
		m_bbTxtSorry.setImage(m_images[IMG_TXT_SORRY]);
		m_bbTxtSorry.moveObject((640 - m_bbTxtSorry.width)/2, TXT_SORRY_Y);
		
		m_bbTxtIntro = new BackBuffer();
		m_bbTxtIntro.createForImage(m_component, m_imob);
		m_bbTxtIntro.setImage(m_images[IMG_TXT_INTRO]);
		m_bbTxtIntro.moveObject((640 - m_bbTxtIntro.width)/2, TXT_INTRO_Y);
		
		// Prepare mouse object
		m_bbMouse = new BackBuffer();
		m_bbMouse.createForImage(m_component, m_imob);
		m_bbMouse.setImage(m_images[IMG_MOUSE]);

		setStateIntro();
	}
	
	/** PAINT HANDLING *****************************************************/	

	boolean __PAINT_HANDLING____________;		// just for nicer view in Class Outline
	
	String TXT_INITIALIZING = new String("Initializing...");
					  
	public void paint(Graphics g, int x, int y)
	{
		if (m_iState == STATE_INITIALIZING)
		{
			g.fillRect(0, 0, 640, 480);
			g.drawString(TXT_INITIALIZING, 100, 100);
		}
		else
		{
			if (m_currentBackBuffer != null)
			{
				m_currentBackBuffer.paint(g, x, y);
			}
			else
			{
				g.fillRect(0, 0, 640, 480);
				g.drawString("Internal error - no game selected...", 100, 100);
			}
		}
	}
	
	public void update(Graphics g, int x, int y)
	{
		if (m_iState == STATE_INITIALIZING)
		{
			g.fillRect(0, 0, 640, 480);
			g.drawString(TXT_INITIALIZING, 100, 100);
		}
		else
		{
			if (m_currentBackBuffer != null)
			{
				m_currentBackBuffer.update(g, x, y);
			}
			else
			{
				g.fillRect(0, 0, 640, 480);
				g.drawString("Internal error - no game selected...", 100, 100);
			}
		}
	}
	
	/** GAME MANIPULATION **************************************************/

	boolean __GAME_MANIPULATION__________;	// just for nice view in Class Outline
	
	final static int GAME_HOTEL		= 1;
	final static int GAME_FIGHT		= 2;
	
	int m_iGame = GAME_HOTEL;				// current (sub)game
	
	ActiveBackBuffer	m_currentBackBuffer;
	
	public void setGameHotel()
	{
		m_currentBackBuffer = m_hotel;
		m_iGame = GAME_HOTEL;
		m_currentBackBuffer.invalidate();
	}
	
	public void setGameFight()
	{
		m_currentBackBuffer = m_fight;
		m_iGame = GAME_FIGHT;
		m_currentBackBuffer.invalidate();
	}
	
	
	/** STATE HANDLING *****************************************************/

	boolean __STATE_MANIPULATION________;	// just for nice view in Class Outline
	
	/** GAME STATES ***********************************************************/
	
	final static int STATE_INITIALIZING				= 0;
	final static int STATE_INTRO					= 1;
	final static int STATE_GET_READY				= 2;
	final static int STATE_PLAYING					= 3;
	final static int STATE_EXIT_GAME				= 4;	
	final static int STATE_GAME_OVER				= 5;
	final static int STATE_PREPARE_FOR_BONUS_GAME	= 6;
	final static int STATE_LIFE_LESS				= 7;
	final static int STATE_END_OF_FIGHT_GAME		= 8;

	int m_iState = STATE_INITIALIZING;

	// Menu data
	
	final static int MENU_ITEM_RESTART_GAME		= 1;
	final static int MENU_ITEM_CONTINUE_GAME	= 2;
	int m_iCurrentMenuItem = MENU_ITEM_CONTINUE_GAME;
	
	public void setStateIntro()
	{
		m_iState = STATE_INTRO;
		m_iMouseCursor = MOUSE_CURSOR_HAND;
		removeAllTexts();
		
		m_hotel.prepareForNewGame();
		m_iLevel = 1;
		m_hotel.prepareForLevel(m_iLevel);
		setGameHotel();
		
		removeAllTexts();
		m_currentBackBuffer.addObject(m_bbTxtIntro);
		m_currentBackBuffer.showObject(m_bbTxtIntro);
	}
	
	public void setStateGetReady()
	{
		m_iState = STATE_GET_READY;
		m_iMouseCursor = MOUSE_CURSOR_ARROW;
		removeAllTexts();
		m_currentBackBuffer.addObject(m_bbTxtGetReady);
		m_currentBackBuffer.showObject(m_bbTxtGetReady);
		m_blinker.reset();
	}
	
	public void setStatePlaying()
	{
		m_iState = STATE_PLAYING;
		m_iMouseCursor = MOUSE_CURSOR_CROSS;
		removeAllTexts();
		m_bMouseClicked = false;
		m_bKeyEsc = false;
		m_currentBackBuffer.addObject(m_bbMouse);
	}
	
	public void setStatePrepareForBonusGame()
	{
		m_iState = STATE_PREPARE_FOR_BONUS_GAME;
		m_iMouseCursor = MOUSE_CURSOR_ARROW;
		
		removeAllTexts();
		m_currentBackBuffer.addObject(m_bbTxtPrepareForBonusGame);
		m_currentBackBuffer.showObject(m_bbTxtPrepareForBonusGame);
		m_blinker.reset();
		
		m_stateToggler.m_lTogglePeriod = 3000;
		m_stateToggler.reset();		
	}
	
	public void setStateLifeLess()
	{
		m_iState = STATE_LIFE_LESS;
		m_iMouseCursor = MOUSE_CURSOR_ARROW;
		
		removeAllTexts();
		m_currentBackBuffer.addObject(m_bbTxtSorry);
		m_currentBackBuffer.showObject(m_bbTxtSorry);
		m_blinker.reset();
		
		m_stateToggler.m_lTogglePeriod = 5000;
		m_stateToggler.reset();		// when this will change, "demo" will stop..
	}
	
	public void setStateGameOver()
	{
		m_iState = STATE_GAME_OVER;
		
		removeAllTexts();
		m_iMouseCursor = MOUSE_CURSOR_ARROW;
		m_currentBackBuffer.addObject(m_bbTxtGameOver);
		m_currentBackBuffer.showObject(m_bbTxtGameOver);
		m_blinker.reset();
		
		m_stateToggler.m_lTogglePeriod = 2000;
		m_stateToggler.reset();		// when this will change, "demo" will be ready for user click to back to menu..
	}

	public void setStateEndOfFightGame()
	{
		m_iState = STATE_END_OF_FIGHT_GAME;
		
		removeAllTexts();
		m_iMouseCursor = MOUSE_CURSOR_ARROW;
		
		m_stateToggler.m_lTogglePeriod = 5000;
		m_stateToggler.reset();		// when this will change, "demo" will be ready for user click to back to menu..
	}

	public void setStateExitGame()
	{
		m_iState = STATE_EXIT_GAME;
		
		removeAllTexts();
		m_iMouseCursor = MOUSE_CURSOR_ARROW;
		m_currentBackBuffer.addObject(m_bbTxtExitGame);
		m_currentBackBuffer.showObject(m_bbTxtExitGame);
		m_blinker.reset();
		
		m_stateToggler.m_lTogglePeriod = 1000;
		m_stateToggler.reset();		// when this will change, "demo" will be ready for user click to back to menu..
	}
	
	
	/**
	 * Main action routine - called periodically by e.g. applet's thread.
	 * After this, applet will usually call update() to show new stuff on screen.
	 * @param long lTimeForAction	 - time passed since last call to this action()
	 */
	public void action(long lTimeForAction)
	{
		// get mouse and keyboard indicators to process..
		m_iMouseX = m_iNewMouseX;
		m_iMouseY = m_iNewMouseY;
		m_bMouseClicked = m_bNewMouseClicked;
		
		m_bKeyEsc = m_bNewKeyEsc;
		
		// clear indicators
		m_bNewMouseClicked = false;
		m_bNewKeyEsc = false;

		m_bbMouse.moveObject(m_iMouseX - m_bbMouse.width/2, m_iMouseY - m_bbMouse.height/2 + 1);
		
		// while there is time to spend
		while(true)
		{
			//
			// Give time to handler for current state
			// (and handler will return time left after its action)
			//
			
			if (m_iState == STATE_INITIALIZING)
				return;
			else if (m_iState == STATE_INTRO)
				lTimeForAction = doStateIntro(lTimeForAction);
			else if (m_iState == STATE_GET_READY)
				lTimeForAction = doStateGetReady(lTimeForAction);
			else if (m_iState == STATE_PLAYING)
				lTimeForAction = doStatePlaying(lTimeForAction);
			else if (m_iState == STATE_EXIT_GAME)
				lTimeForAction = doStateExitGame(lTimeForAction);
			else if (m_iState == STATE_GAME_OVER)
				lTimeForAction = doStateGameOver(lTimeForAction);
			else if (m_iState == STATE_PREPARE_FOR_BONUS_GAME)
				lTimeForAction = doStatePrepareForBonusGame(lTimeForAction);
			else if (m_iState == STATE_LIFE_LESS)
				lTimeForAction = doStateLifeLess(lTimeForAction);
			else if (m_iState == STATE_END_OF_FIGHT_GAME)
				lTimeForAction = doStateEndOfFightGame(lTimeForAction);
			else
				lTimeForAction = 0;
			
			if (lTimeForAction <= 0)
				break;
			
			m_bMouseClicked = false;
		}
	}
	

	public long doStateIntro(long lTimeForAction)
	{
		if ((m_bMouseClicked) || (m_bKeyEsc))
		{
			m_bMouseClicked = false;
			m_bKeyEsc = false;
			setStateGetReady();
			return lTimeForAction;
		}
		else
		{
			return 0;
		}
	}

	/**
	 * - if mouse clicked
	 *		- continues playing (activates STATE_PLAYING)
	 * - else if esc clicked
	 *		- activates STATE_MENU
	 * - else
	 *		- blinks text (continue waiting for user action)
	 * 
	 * @return time left after processing action
	 */
	public long doStateGetReady(long lTimeForAction)
	{
		if (m_bMouseClicked)
		{
			m_bMouseClicked = false;
			setStatePlaying();
			return lTimeForAction;
		}
		else if (m_bKeyEsc)
		{
			m_bKeyEsc = false;
			m_currentBackBuffer.removeObject(m_bbTxtGetReady);
			setStateExitGame();
			return lTimeForAction;
		}
		else
		{
			m_blinker.doAction(lTimeForAction, m_bbTxtGetReady, m_currentBackBuffer);
			return 0;
		}
	}

	public long doStatePrepareForBonusGame(long lTimeForAction)
	{
		// if toggler has not yet changed (it has m_bChanged = true and m_bFlag = true
		if (!m_stateToggler.m_bChanged)
		{
			m_blinker.doAction(lTimeForAction, m_bbTxtPrepareForBonusGame, m_currentBackBuffer);
			m_stateToggler.doAction(lTimeForAction);
			doPlayHotelInDemoMode(lTimeForAction);
			return 0;
		}
		// when toggler has changed..
		else
		{
			removeAllTexts();
						
			// switch to fight game
			m_fight.prepareForGame(m_iLevel, m_hotel.m_iLives);
			setGameFight();
			removeAllTexts();
			m_currentBackBuffer.addObject(m_bbMouse);
			setStateGetReady();
			return 0;
		}
	}

	public long doStateLifeLess(long lTimeForAction)
	{
		// blink 'Sorry'
		m_blinker.doAction(lTimeForAction, m_bbTxtSorry, m_currentBackBuffer);
		
		// check to end demo
		m_stateToggler.doAction(lTimeForAction);
		if (m_stateToggler.m_bChanged)
		{
			m_hotel.prepareForLevel(m_iLevel);
			setStateGetReady();
			return 0;
		}
		
		doPlayHotelInDemoMode(lTimeForAction);
		return 0;
	}

	
	public long doStateGameOver(long lTimeForAction)
	{
		// if toggler has not yet changed (it has m_bChanged = true and m_bFlag = true
		if (!m_stateToggler.m_bChanged)
		{
			m_blinker.doAction(lTimeForAction, m_bbTxtGameOver, m_currentBackBuffer);
			m_stateToggler.doAction(lTimeForAction);
			doPlayHotelInDemoMode(lTimeForAction);
		}
		// when toggler has changed..
		else
		{
			// if just toggled (toggled to false)
			if (!m_stateToggler.m_bFlag)
			{
				m_hotel.prepareForLevel(m_iLevel);
				m_stateToggler.m_bFlag = true;
			}
			else
			{
				// wait for mouse click or esc to go to intro
				if ((m_bMouseClicked) || (m_bKeyEsc))
				{
					m_bMouseClicked = false;
					m_bKeyEsc = false;
					setStateIntro();
				}
			}
		}
		return 0;
	}

	public long doStateEndOfFightGame(long lTimeForAction)
	{
		m_fight.action(lTimeForAction);
		m_stateToggler.doAction(lTimeForAction);
		
		// check for sound effects
		if (m_fight.m_bSndPlayerGun)
		{
			m_fight.m_bSndPlayerGun = false;
			m_bSndFightGun = true;
		}
		else if (m_fight.m_bSndPlayerGunHat)
		{
			m_fight.m_bSndPlayerGunHat = false;
			m_bSndFightGun = true;
		}
		
		if (m_stateToggler.m_bChanged)
		{
			// add lives from fight game to hotel game
			m_hotel.m_iLives += m_fight.m_iNewLives;
			
			// increase level
			m_iLevel++;
			if (m_iLevel == 100)
				m_iLevel = 1;		// sorry :-)
			
			// switch back to hotel game...
			m_hotel.prepareForLevel(m_iLevel);
			setGameHotel();
			setStateGetReady();
		}
		return 0;
	}
	
	public long doStatePlaying(long lTimeForAction)
	{
		if (m_bKeyEsc)
		{
			m_bKeyEsc = false;
			setStateExitGame();
			return lTimeForAction;
		}
		else
		{
			while(m_iState == STATE_PLAYING)
			{
				if (m_iGame == GAME_HOTEL)
					lTimeForAction = doPlayHotel(lTimeForAction);
				else if (m_iGame == GAME_FIGHT)
					lTimeForAction = doPlayFight(lTimeForAction);
				else lTimeForAction = 0;
	
				if (lTimeForAction <= 0)
					break;
			}
			return lTimeForAction;
		}
	}
	
	public long doStateExitGame(long lTimeForAction)
	{
		// if toggler has not yet changed (it has m_bChanged = true and m_bFlag = true
		if (!m_stateToggler.m_bChanged)
		{
			m_blinker.doAction(lTimeForAction, m_bbTxtExitGame, m_currentBackBuffer);
			m_stateToggler.doAction(lTimeForAction);
			return 0;
		}
		// when toggler has changed..
		else
		{
			// if just toggled (toggled to false)
			if (!m_stateToggler.m_bFlag)
			{
				m_currentBackBuffer.showObject(m_bbTxtExitGame);
				m_stateToggler.m_bFlag = true;
			}
			else
			{
				if (m_bMouseClicked)
				{
					m_bMouseClicked = false;
					setStatePlaying();
					return 0;
				}
				else if (m_bKeyEsc)
				{
					m_bKeyEsc = false;
					removeAllTexts();
					setStateIntro();
					return 0;
				}
			}
		}
		
		return 0;
	}


	public long doPlayHotel(long lTimeForAction)
	{
		m_hotel.m_iMouseX = m_iMouseX;
		m_hotel.m_iMouseY = m_iMouseY;
		m_hotel.m_bMouseClicked = m_bMouseClicked;
		
		if ((m_iState == STATE_PLAYING) && (m_iGame == GAME_HOTEL))
		{
			// if hotel game is requesting that player goes to next level..
			if (m_hotel.m_iResponse == m_hotel.RESPONSE_READY_FOR_NEXT_LEVEL)
			{
				setStatePrepareForBonusGame();
				return 0;
			}
			else if (m_hotel.m_iResponse == m_hotel.RESPONSE_READY_FOR_LIFE_LESS)
			{
				setStateLifeLess();
				return 0;
			}
			else if (m_hotel.m_iResponse == m_hotel.RESPONSE_READY_FOR_GAME_OVER)
			{
				setStateGameOver();
				return 0;
			}
			else
			{
				m_hotel.action(lTimeForAction);
				
				// check for sound effects
				if (m_hotel.m_bSndPlayerGun)
				{
					m_hotel.m_bSndPlayerGun = false;
					m_bSndPlayerGun = true;
				}
				else if (m_hotel.m_bSndPlayerGunHat)
				{
					m_hotel.m_bSndPlayerGunHat = false;
					m_bSndPlayerGunHat = true;
				}
				return 0;
			}
		}
		
		return 0;
	}
	
	public long doPlayFight(long lTimeForAction)
	{
		m_fight.m_iMouseX = m_iMouseX;
		m_fight.m_iMouseY = m_iMouseY;
		m_fight.m_bMouseClicked = m_bMouseClicked;
		
		if ((m_iState == STATE_PLAYING) && (m_iGame == GAME_FIGHT))
		{
			// if fight game is requesting that player goes to next level..
			if (m_fight.m_iResponse == m_fight.RESPONSE_END_OF_GAME)
			{
				setStateEndOfFightGame();
				return 0;
			}
			
			m_fight.action(lTimeForAction);
				
			// check for sound effects
			if (m_fight.m_bSndPlayerGun)
			{
				m_fight.m_bSndPlayerGun = false;
				m_bSndFightGun = true;
			}
			else if (m_fight.m_bSndPlayerGunHat)
			{
				m_fight.m_bSndPlayerGunHat = false;
				m_bSndFightGunHat = true;
			}
		}
		
		return 0;
	}

	
	private void doPlayHotelInDemoMode(long lTimeForAction)
	{
		if (m_iGame == GAME_HOTEL)
			m_hotel.action(lTimeForAction);
	}
	
	private void removeAllTexts()
	{
		if (m_currentBackBuffer != null)
		{
			m_currentBackBuffer.removeObject(m_bbTxtGetReady);
			m_currentBackBuffer.removeObject(m_bbTxtGameOver);
			m_currentBackBuffer.removeObject(m_bbTxtExitGame);
			m_currentBackBuffer.removeObject(m_bbTxtPrepareForBonusGame);
			m_currentBackBuffer.removeObject(m_bbTxtSorry);
			m_currentBackBuffer.removeObject(m_bbTxtIntro);
			m_currentBackBuffer.removeObject(m_bbMouse);
		}
	}
	
	/** MOUSE HANDLING *****************************************************/

	boolean __MOUSE_HANDLING____________;		// just for nicer view in Class Outline

	// mouse cursors
	final static int MOUSE_CURSOR_ARROW = 1;
	final static int MOUSE_CURSOR_HAND  = 2;
	final static int MOUSE_CURSOR_CROSS = 3;
	
	int m_iMouseCursor = MOUSE_CURSOR_ARROW;
	
	// Current mouse indicators (ones to be processed)
	// (these ones are updated by onMouseMoved() & onMouseClicked() handlers)
	private int		m_iNewMouseX;
	private int		m_iNewMouseY;
	private boolean	m_bNewMouseClicked;		// true when clicked

	// Mouse indicators that are currently processed
	// (these ones are filled by action() routine, and then proccesed by onState functions)
	private int		m_iMouseX;
	private int		m_iMouseY;
	private boolean	m_bMouseClicked;		// true when clicked

	/**
	 * Call when mouse has moved
	 */
	public void onMouseMoved(int x, int y)
	{
		m_iNewMouseX = x;
		m_iNewMouseY = y;
	}
	
	/**
	 * Call when mouse is clicked
	 */
	public void onMouseClicked(int x, int y)
	{
		m_iNewMouseX = x;
		m_iNewMouseY = y;
		m_bNewMouseClicked = true;
	}
	
	/** KEYBOARD HANDLING **************************************************/

	boolean __KEYBOARD_HANDLING_______;		// just for nicer view in Class Outline

	// current keyboard indicators (ones to be processed)
	private boolean	m_bNewKeyEsc = false;
	
	// keyboard indicators that are currently processed
	private boolean	m_bKeyEsc = false;

	public void onKeyEsc()
	{
		m_bNewKeyEsc = true;
	}
}
