/**
 *
 * Door.java
 * 
 * Project:	Disney Hotel Game
 *
 * Author:	Matija Tomaskovic
 *
 *** DESCRIPTION & USAGE ***
 *		
 * This class represents door sprite on some container (like ShootArea).
 * 
 * To use it, do following:
 * 
 * 1. Create instance of this object:
 *	
 *		Door door = new Door(imgDoorClosed, imgDoorOpened, 5
 *							yourAppletComponent, yourImageObserver)
 * 
 * 2. Additional initialization
 * 
 *		door.setOpeningFrame(0, frame1Image, 300);
 *		door.setOpeningFrame(1, frame2Image, 500);
 *		:
 *		max. is 5 because of parameter passed in constructor!
 * 
 *	(do not skip frames!, e.g. creating frame 0 and 2, but skiping 1)
 * 
 * 3. Action:
 * 
 *		door.spendTime(300);
 * 
 * Check STATE_* for door states...
 *
 *** LOG ***
 * 
 * 23.Dec.1999.
 * ~~~~~~~~~~~~
 *	10:27..	- created
 *
 */

import java.awt.*;
import java.awt.image.ImageObserver;
import java.lang.Math;



public class Door
	extends ActiveBackBuffer
{
	/** PARENT DATA ********************************************************/

	Component		m_component;
	ImageObserver	m_imob;
	
	/** DOOR STATES ********************************************************/
	
	final static int STATE_CLOSED		= 1;
	final static int STATE_CAN_OPEN		= 2;
	final static int STATE_OPENING		= 3;
	final static int STATE_OPENED		= 4;	// door stucks here (HotelGame should switch to STATE_CAN_CLOSE)
	final static int STATE_CAN_CLOSE	= 5;
	final static int STATE_CLOSING		= 6;

	int m_iState;
	
	/** DOOR FRAMES ***********************************************************/

	Image m_imgDoorClosed;
	Image m_imgDoorOpened;

	Image	m_OpeningImageArray[];
	long	m_OpeningTimeArray[];
	int		m_iImgOpeningArraySize;
	int		m_iImgOpeningFrames;		// number of opening frames.. (0, 1, 2,..)
	
	int		m_iCurrentOpeningFrame;		// current opening frame (0, 1,..) when STATE_OPENING

	/** INITIALIZATION *****************************************************/

	/**
	 * Construction
	 */
	public Door(Image imgDoorClosed, Image imgDoorOpened,
				int iMaxOpeningFrames, 				
				Component c, ImageObserver imob)
	{
		// initialize BackBuffer
		createForImage(c, imob);
		
		// remember parent data
		m_component = c;
		m_imob = imob;
		
		// remember 'closed' and 'opening' image
		m_imgDoorClosed = imgDoorClosed;
		m_imgDoorOpened = imgDoorOpened;
		
		// prepare for opening frames
		m_iImgOpeningArraySize = iMaxOpeningFrames;
		m_OpeningImageArray = new Image[m_iImgOpeningArraySize];
		m_OpeningTimeArray = new long[m_iImgOpeningArraySize];
		
		// set to closed state...
		setStateClosed();
	}
	
	/**
	 * Sets frame image and delay for frame (time to wait for that frame)
	 */
	public void setOpeningFrame(int iFrame, Image imgFrame, long lDelay)
	{
		// if array to small - ignore
		if (iFrame >= m_iImgOpeningArraySize)
			return;
		
		m_OpeningImageArray[iFrame] = imgFrame;
		m_OpeningTimeArray[iFrame] = lDelay;
		
		if (iFrame >= m_iImgOpeningFrames)
			m_iImgOpeningFrames = iFrame + 1;
	}

	
	/** ACTION *************************************************************/

	public void setStateClosed()
	{
		m_iState = STATE_CLOSED;
		setImage(m_imgDoorClosed);
		setActionNone();
	}
	
	public void setStateOpened()
	{
		m_iState = STATE_OPENED;
		setImage(m_imgDoorOpened);
		setActionNone();
	}
	
	public void setStateCanOpen()
	{
		m_iState = STATE_CAN_OPEN;
		setActionNext();
	}
	
	public void setStateCanClose()
	{
		m_iState = STATE_CAN_CLOSE;
		setActionNext();
	}
	
	
	/**
	 * This will do next action, and set
	 */
	public void doNextAction()
	{
		if (m_iState == STATE_CLOSED)
		{
			// do nothing, parent will switch to STATE_CAN_OPEN
			setActionNone();
		}	
		else if (m_iState == STATE_CAN_OPEN)
		{
			// start opening...
			
			m_iCurrentOpeningFrame = 0;
			
			// if no more opening frames
			if (m_iCurrentOpeningFrame >= m_iImgOpeningFrames)
			{ 
				setStateOpened();
				return;
			}
			else
			{
				// set opening frame
				setImage(m_OpeningImageArray[m_iCurrentOpeningFrame]);
				setActionWhenTime(m_OpeningTimeArray[m_iCurrentOpeningFrame]);

				m_iState = STATE_OPENING;
			}
		}
		else if (m_iState == STATE_OPENING)
		{
			m_iCurrentOpeningFrame++;
			
			// if no more opening frames
			if (m_iCurrentOpeningFrame >= m_iImgOpeningFrames)
			{
				setStateOpened();
				return;
			}
			else
			{
				// set opening frame
				setImage(m_OpeningImageArray[m_iCurrentOpeningFrame]);
				setActionWhenTime(m_OpeningTimeArray[m_iCurrentOpeningFrame]);
			}
		}
		else if (m_iState == STATE_OPENED)
		{
			// do nothing - HotelGame (parent) should switch to STATE_CAN_CLOSE
			setActionNone();
		}
		else if (m_iState == STATE_CAN_CLOSE)
		{
			// start closing..
			m_iState = STATE_CLOSING;
			m_iCurrentOpeningFrame = m_iImgOpeningFrames - 1;
			
			// set closing frame
			setImage(m_OpeningImageArray[m_iCurrentOpeningFrame]);
			setActionWhenTime(m_OpeningTimeArray[m_iCurrentOpeningFrame]);
		}
		else if (m_iState == STATE_CLOSING)
		{
			// do next frame..
			m_iCurrentOpeningFrame--;
					
			// if no more opening frames..
			if (m_iCurrentOpeningFrame < 0)
			{
				setStateClosed();
				return;
			}
			
			// set opening frame
			setImage(m_OpeningImageArray[m_iCurrentOpeningFrame]);
			setActionWhenTime(m_OpeningTimeArray[m_iCurrentOpeningFrame]);
		}
	}
}
