/**
 *
 * GoodGuy.java
 * v1.0
 * 
 * Project:	Coral Game
 *
 * by Matija Tomaskovic
 * (mataya@unforgettable.com || http://www.foi.hr/~mtomasko)
 *
 *** DESCRIPTION & USAGE ***
 *		
 * This class represents Good Guy. This one is usually in some
 * shoot area.
 * 
 * To use it, do following:
 * 
 * 1. Create it, and initialize it:
 * 
 * 		GoodGuy	guy = new GoodGuy(m_imgGoodGuyStanding, m_imgGoodGuyDead, 4, m_component, m_imob);
 * 		guy.setDyingFrame(0, m_imgBadGuyDying1, 0, 0, 100);
 * 		guy.setDyingFrame(1, m_imgBadGuyDying2, 0, 0, 100);
 * 		guy.setDyingFrame(2, m_imgBadGuyDying3, 0, 0, 100);
 * 		guy.setDyingFrame(3, m_imgBadGuyDying4, 0, 0, 100);
 * 		guy.m_lMinStandingDelay = 500;
 * 		guy.m_lMinStandingDelay = 2000;
 * 
 * 		guy.moveObject(100, 200);
 * 
 * 2. Optionally, give him hat:
 * 
 * 		guy.setHat(imgHat, 90, 170, imgHatDown, imgHatDownScore);
 * 
 * 3. Activate it - guy will add himself (and his hat) to parent:
 * 
 *		guy.activate(parent);
 * 
 * 4. Here, guy is doing his job... because parent timing/action
 *	  mechanism is working in the background.
 *	  
 *    First, guy will be in STATE_STANDING, waiting some time.
 *    The, guy will go to STATE_WAIT_TO_CLOSE_DOOR and set RESPONSE_CLOSE_DOOR
 *    (asking from his parent that door should be closed).
 * 
 *    Meantime, you can call 
 * 
 *		guy.onMouseClick(x, y);
 * 
 *    ..that can shoot the guy or his hat (then, guy will set 
 *    RESPONSE_HAT_IS_SHOT or RESPONSE_GUY_IS_SHOOT). 
 *    If guy is shot, he will first do STATE_DYING, and then STATE_DEAD.
 *
 *  5. At the end call: (guy will remove itself and his hat from parent)
 * 
 *		guy.dismiss(parent);
 *
 *** LOG ***
 * 
 * 23.Dec.1999.
 * ~~~~~~~~~~~~
 *	15:18..	- created
 *
 * 30.Dec.1999.
 * ~~~~~~~~~~~~
 *	16:29..	- finalized v1.0
 */
import java.awt.*;
import java.awt.image.ImageObserver;
import java.awt.image.PixelGrabber;
import java.lang.Math;


public class GoodGuy
	extends ActiveBackBuffer
{
	/** IMAGES *************************************************************/

	Image	m_imgGuyStanding;
	Image	m_imgGuyDead;
	
	ImageFrames	m_GuyDyingFrames;
	

	/** STATES *************************************************************/
	
	final static int STATE_STANDING			= 1;	// guy is standing (showing up to player)
	final static int STATE_WAIT_TO_CLOSE_DOOR = 2;	// guy started waiting for door to be closed
	final static int STATE_DYING			= 3;	// guy was just shoot and is dying...
	final static int STATE_DEAD				= 4;	// guy was shoot and is lying dead
	
	int m_iState;

	final static int RESPONSE_NONE			= 0;
	final static int RESPONSE_CLOSE_DOOR	= 1;	// this is response that guy asks door to be closed
	final static int RESPONSE_HAT_IS_SHOT	= 2;	// notification that hat is just shoot
	final static int RESPONSE_GUY_IS_SHOT	= 3;	// notification that guy is just shoot
	final static int RESPONSE_GUY_IS_DEAD	= 4;	// notification that guy is lying dead

	int m_iResponse = RESPONSE_NONE;
	
	Hat	m_hat;
	
	
	/** CONSTRUCTION *******************************************************/

	public GoodGuy(Image imgGuyStanding, Image imgGuyDead, 
				  int iMaxDyingFrames,
				  Component c, ImageObserver imob)
	{
		createForImage(c, imob);
		m_strName = "BADGUY";
		
		// remember given images
		m_imgGuyStanding = imgGuyStanding;
		m_imgGuyDead = imgGuyDead;
		
		m_GuyDyingFrames = new ImageFrames(iMaxDyingFrames);
		
		setStateStanding();
	}

	/** INITIALIZATION *****************************************************/
	
	public void setDyingFrame(int iFrame, Image imgFrame, int x, int y, long lDelay)
	{
		m_GuyDyingFrames.setFrame(iFrame, imgFrame, x, y, lDelay);
	}
	
	public void setHat(Image imgHat, int x, int y, Image imgHatDown, Image imgHatDownScore)
	{
		m_hat = new Hat(imgHat, x, y, 
						imgHatDown, this.y + m_imgGuyStanding.getHeight(m_imob) - imgHatDown.getHeight(m_imob), 
						imgHatDownScore,
						m_component, m_imob);
	}
	
	
	/** ACTION *************************************************************/
	
	long m_lMinStandingDelay = 2000;	// min. delay (in milliseconds) to wait to start Standing
	long m_lMaxStandingDelay = 4000;	// max. delay (in milliseconds) to wait to start Standing

	public void setStateStanding()
	{
		m_iState = STATE_STANDING;
		m_iResponse = RESPONSE_NONE;
		setImage(m_imgGuyStanding);
		
		// calculate delay until closing door
	
		double d = m_lMinStandingDelay + (Math.random() * (m_lMaxStandingDelay - m_lMinStandingDelay));
		Double dd = new Double(d);
		
		setActionWhenTime(dd.longValue());
	}
	
	public void setStateDying()
	{
		m_iState = STATE_DYING;		
		m_iResponse = RESPONSE_GUY_IS_SHOT;
		
		// if there is hat..
		if (m_hat != null)
		{
			if (m_hat.m_iState == m_hat.STATE_ON_HEAD)
				m_hat.setStateShooted();
		}
		
		m_GuyDyingFrames.m_iCurrentFrame = 0;
		Image img = m_GuyDyingFrames.getCurrentImage();
		if (img != null)
		{
			setImage(img);
			setActionWhenTime(m_GuyDyingFrames.getCurrentDelay());
		}
		else
		{
			setStateDead();
		}
	}
	
	public void setStateDead()
	{
		setImage(m_imgGuyDead);
		m_iState = STATE_DEAD;
		setActionNone();
	}
	
	
	/**
	 * This will do next action, and set
	 */
	public void doNextAction()
	{
		if (m_iState == STATE_STANDING)
			doStateStanding();
		else if (m_iState == STATE_DYING)
			doStateDying();
		else if (m_iState == STATE_DEAD)
			doStateDead();
	}
	
	private void doStateStanding()
	{
		m_iState = STATE_WAIT_TO_CLOSE_DOOR;
		m_iResponse = RESPONSE_CLOSE_DOOR;
		setActionNone();
	}
	
	private void doStateDying()
	{
		// do next frame..
		if (m_GuyDyingFrames.nextFrame())
		{
			setImage(m_GuyDyingFrames.getCurrentImage());
			setActionWhenTime(m_GuyDyingFrames.getCurrentDelay());
		}
		// if no more dying frames..
		else
		{
			setStateDead();
		}
	}
	
	private void doStateDead()
	{
		setActionNone();
	}

	
	public void activate(ActiveBackBuffer parent)
	{
		parent.addObject(this);
		parent.moveObjectToBack(this);
		
		// check to add hat
		if (m_hat != null)
		{
			m_hat.setStateOnHead();
			parent.removeObject(m_hat);
			parent.addObject(m_hat);
			m_hat.moveObject(m_hat.m_iHatX, m_hat.m_iHatY);
			parent.moveObjectToBack(m_hat);
		}
		
		setStateStanding();
	}

	
	public void dismiss(ActiveBackBuffer parent)
	{
		if (m_hat != null)
		{
			parent.removeObject(m_hat);
		}
					
		parent.removeObject(this);
	}
	
	
	public void onMouseClick(int x, int y)
	{
		if ((m_iState == STATE_STANDING) ||	(m_iState == STATE_WAIT_TO_CLOSE_DOOR))
		{
			if (hasPixelAt(x, y))
			{
				setStateDying();
			}
			else
			{
				if (m_hat != null)
				{
					if (m_hat.m_iState == m_hat.STATE_ON_HEAD)
					{
						if (m_hat.hasPixelAt(x, y))
						{
							m_hat.setStateShootedWithScore();
							m_iResponse = RESPONSE_HAT_IS_SHOT;
						}
					}
				}
			}
		}
	}
}
