/**
 * HotelGame.java - main engine class
 * v1.0
 * 
 * Project: Coral game
 *
 * by Matija Tomaskovic
 * (mataya@unforgettable.com || http://www.foi.hr/~mtomasko)
 *
 *** DESCRIPTION AND USAGE ***
 * 
 * HotelGame represents main Hotel Game engine.
 * 
 * To use it, do following:
 * 
 * 1. Create it:
 * 
 * 		m_hotel = new HotelGame(m_component, m_imob);
 * 
 * 2. Assign it all pictures (see m_img*)
 * 
 * 		m_hotel.m_imgBackground = m_images[IMG_BACKGROUND];
 * 		m_hotel.m_imgLife = m_images[IMG_LIFE];
 *		:
 *		(check m_img*)
 * 
 * 3. Initialize it
 * 
 *		m_hotel.init();
 * 
 * 4. Prepare for new game:
 *		
 *		m_hotel.prepareForNewGame();		// Score = 0; Lives = INITIAL_NUMBER_OF_LIVES;..
 * 
 * 5. Prepare for level
 *		
 *		m_hotel.prepareForLevel(1);
 * 
 * 6. Do in loop:
 * 
 * 		m_hotel.m_iMouseX = m_iMouseX;
 * 		m_hotel.m_iMouseY = m_iMouseY;
 * 		m_hotel.m_bMouseClicked = m_bMouseClicked;
 * 
 *		m_hotel.action(lTimeForAction);
 * 
 * 		// check for sound effects
 * 		if (m_hotel.m_bSndPlayerGun)
 * 		{
 * 			m_hotel.m_bSndPlayerGun = false;
 *			...
 * 		}
 * 		else if (m_hotel.m_bSndPlayerGunHat)
 * 		{
 * 			m_hotel.m_bSndPlayerGunHat = false;
 *			...
 * 		}
 * 
 *		// check for responses...
 * 
 *		// if ...
 *			// go to step 5. (new level)
 * 
 *** LOG ***
 * 
 * 08.Aug.1999.
 * ~~~~~~~~~~~~
 *	- created
 * 
 * 21.Dec.1999.
 * ~~~~~~~~~~~~
 *	22:32..	- redesigning...
 *	..00:39	- STATE_* added.. 
 *			- finished: STATE_INITIALIZING, STATE_GET_READY, 
 *			  STATE_CHECK_TO_EXIT_GAME,	STATE_GAME_OVER, and shell for STATE_PLAYING
 *	..01:14 - finished painting of lives
 *	..01:24	- finished painting bullets
 *	..01:46 - finished painting scores
 *			- "Get Ready!" is now blinking..
 * 
 * 02.Jan.1999.
 * ~~~~~~~~~~~~
 *	15:37..	- v1.0 - finished!
 */

import java.awt.*;
import java.awt.image.*;
import java.awt.image.ImageObserver;
import java.util.Vector;

public class HotelGame
	extends ActiveBackBuffer
{
	/** PARENT DATA ********************************************************/
	
	ImageObserver	m_imob;
	Component		m_component;
	
	/** IMAGES (must be given by creator of this class) ********************/
	
	Image m_imgBackground;
	Image m_imgLife;
	
	Image m_imgNumberNull;
	Image m_imgNumber0;
	Image m_imgNumber1;
	Image m_imgNumber2;
	Image m_imgNumber3;
	Image m_imgNumber4;
	Image m_imgNumber5;
	Image m_imgNumber6;
	Image m_imgNumber7;
	Image m_imgNumber8;
	Image m_imgNumber9;
	
	Image m_imgDoorClosed;
	Image m_imgDoorOpened;
	Image m_imgDoorOpening1;
	Image m_imgDoorOpening2;
	Image m_imgDoorOpening3;
	Image m_imgDoorOpening4;

	Image m_imgWindowClosed;
	Image m_imgWindowOpened;
	Image m_imgWindowOpening1;
	Image m_imgWindowOpening2;
	Image m_imgWindowOpening3;
	
	Image m_imgBadGuyStanding;
	Image m_imgBadGuyDead;
	Image m_imgBadGuyDying1;
	Image m_imgBadGuyDying2;
	Image m_imgBadGuyDying3;
	Image m_imgBadGuyDying4;
	Image m_imgBadGuyTriggering1;
	Image m_imgBadGuyTriggering2;
	Image m_imgBadGuyTriggering3;
	Image m_imgBadGuyTriggering4;
	Image m_imgBadGuyShooting1;
	Image m_imgBadGuyShooting2;
	Image m_imgBadGuyShooting3;
	Image m_imgBadGuyShooting4;
	
	Image m_imgGoodGuyStanding;
	Image m_imgGoodBadGuyStanding;
	Image m_imgGoodGuyDead;
	
	Image m_imgWndGoodGuyStanding;
	Image m_imgWndGuyDying1;
	Image m_imgWndGuyDying2;
	Image m_imgWndGuyDying3;
	Image m_imgWndGuyDying4;
	Image m_imgWndGuyDying5;
	Image m_imgWndBadGuyDead;
	Image m_imgWndGoodGuyDead;
	
	Image m_imgWndBadGuyStanding;
	Image m_imgWndBadGuyTriggering1;
	Image m_imgWndBadGuyTriggering2;
	Image m_imgWndBadGuyTriggering3;
	Image m_imgWndBadGuyTriggering4;
	Image m_imgWndBadGuyShooting1;
	Image m_imgWndBadGuyShooting2;
	Image m_imgWndBadGuyShooting3;
	Image m_imgWndBadGuyShooting4;
	
	Image m_imgWndGoodBadGuyStanding;
		
	Image m_imgHat;
	Image m_imgHatDown;
	Image m_imgHatDownScore;
	
	
	/** GAME STATES ********************************************************/
	
	final static int STATE_INITIALIZING			= 0;
	final static int STATE_PLAYING				= 1;
	final static int STATE_READY_FOR_LIFE_LESS	= 2;	// game is running, player input is ignored, all responses are ignored, last killed life (m_iLives+1) is blinking
	final static int STATE_READY_FOR_GAME_OVER	= 3;	// game is running, player input is ignored, all responses are ignored
	final static int STATE_READY_FOR_NEXT_LEVEL	= 4;	// game is running, player input is ignored, all responses are ignored, scores are blinking...
	
	int m_iState = STATE_INITIALIZING;

	final static int RESPONSE_NONE					= 0;
	final static int RESPONSE_READY_FOR_LIFE_LESS	= 1;
	final static int RESPONSE_READY_FOR_NEXT_LEVEL	= 2;
	final static int RESPONSE_READY_FOR_GAME_OVER	= 3;
	
	int m_iResponse = RESPONSE_NONE;
	
	/** DATA ***************************************************************/
	
	final static int LIVES_X				= 25;
	final static int LIVES_Y				= 338;
	
	final static int MAX_SCORE_DIGITS		= 6;
	final static int SCORE_X				= 344;
	final static int SCORE_Y				= 348;

	final static int MAX_LEVEL_DIGITS		= 2;
	final static int LEVEL_X				= 570;
	final static int LEVEL_Y				= 348;											 

	final static int INITIAL_NUMBER_OF_LIVES	= 5;
	final static int MAX_LIVES					= 5;
	
	BackBuffer	m_bbLives[];
	BackBuffer	m_bbLevelDigits[];
	BackBuffer	m_bbScoreDigits[];

	boolean		m_bLastLifeVisible = true;
	boolean		m_bLevelsVisible = true;
	boolean		m_bScoresVisible = true;
	
	/**
	 * GAME PARAMETERS
	 */
	
	int m_iLives;				// current number of lives
	int m_iScore;				// current scores
	int m_iLevel;				// current level
	
	final static int SCORE_FOR_HAT					= 30;
	final static int SCORE_FOR_BAD_GUY_IN_DOOR		= 50;
	final static int SCORE_FOR_BAD_GUY_IN_WINDOW	= 60;

	final static int SCORES_PER_LEVEL				= 300;	// scores needed to pass one level
	final static int SCORES_ADD_PER_LEVEL			= 50;	// scores added to SCORES_PER_LEVEL for each level
	

	// current shoot area list..
	private Vector m_Doors = new Vector(3);
	private Vector m_Windows = new Vector(20);

	private ActionToggle m_toggler = new ActionToggle(250);
	
	/** SOUND EFFECTS FLAGS ************************************************/
	
	boolean m_bSndPlayerGun = false;
	boolean m_bSndPlayerGunHat = false;
	boolean m_bSndBadGuyGun = false;

	
	/** CODE ***************************************************************/
	
	/**
	 * Constructor
	 */
	public HotelGame(Component c, ImageObserver imob)
	{
		createForImage(c, imob);
		
		m_component = c;
		m_imob = imob;
		
		m_iLives = INITIAL_NUMBER_OF_LIVES;
		m_iScore = 0;
	}

	/**
	 * Game initialization.
	 */
	public void init()
	{
		setActionContinuous();
		
		// set background 
		this.setImage(m_imgBackground);

		int i;

		// Prepare objects for lives
		m_bbLives = new BackBuffer[MAX_LIVES];
		for (i = 0; i < MAX_LIVES; i++)
		{
			m_bbLives[i] = new BackBuffer();
			m_bbLives[i].createForImage(m_component, m_imob);
			m_bbLives[i].setImage(m_imgLife);
			m_bbLives[i].moveObject(LIVES_X + i * (m_bbLives[i].width + 1), LIVES_Y);
			addObject(m_bbLives[i]);
		}
		
		// Prepare objects for scores
		m_bbScoreDigits = new BackBuffer[MAX_SCORE_DIGITS];
		for (i = 0; i < MAX_SCORE_DIGITS; i++)
		{
			m_bbScoreDigits[i] = new BackBuffer();
			m_bbScoreDigits[i].createForImage(m_component, m_imob);
			m_bbScoreDigits[i].setImage(m_imgNumberNull);
			m_bbScoreDigits[i].moveObject(SCORE_X + i * m_bbScoreDigits[i].width, SCORE_Y);
			addObject(m_bbScoreDigits[i]);
		}

		// Prepare objects for level
		m_bbLevelDigits = new BackBuffer[MAX_LEVEL_DIGITS];
		for (i = 0; i < MAX_LEVEL_DIGITS; i++)
		{
			m_bbLevelDigits[i] = new BackBuffer();
			m_bbLevelDigits[i].createForImage(m_component, m_imob);
			m_bbLevelDigits[i].setImage(m_imgNumberNull);
			m_bbLevelDigits[i].moveObject(LEVEL_X + i * m_bbLevelDigits[i].width, LEVEL_Y);
			addObject(m_bbLevelDigits[i]);
		}

		// saloon doors
		
		ShootArea sa = CreateRoomDoor();
		sa.moveObject(34, 213 - 0);
		m_Doors.addElement(sa);
		addObject(sa);

		sa = CreateRoomDoor();
		sa.moveObject(267, 213 - 0);
		m_Doors.addElement(sa);
		addObject(sa);

		sa = CreateRoomDoor();
		sa.moveObject(492, 213 - 0);
		m_Doors.addElement(sa);
		addObject(sa);

		// windows
		
		sa = CreateWindow();
		sa.moveObject(34, 107 - 0);
		m_Windows.addElement(sa);
		addObject(sa);
		
		sa = CreateWindow();
		sa.moveObject(131, 107 - 0);
		m_Windows.addElement(sa);
		addObject(sa);
		
		sa = CreateWindow();
		sa.moveObject(229, 107 - 0);
		m_Windows.addElement(sa);
		addObject(sa);

		sa = CreateWindow();
		sa.moveObject(327, 107 - 0);
		m_Windows.addElement(sa);
		addObject(sa);
		
		sa = CreateWindow();
		sa.moveObject(423, 107 - 0);
		m_Windows.addElement(sa);
		addObject(sa);
		
		sa = CreateWindow();
		sa.moveObject(522, 107 - 0);
		m_Windows.addElement(sa);
		addObject(sa);
	}
	
	/**
	 * CreateRoomDoor
	 */
	private ShootArea CreateRoomDoor()
	{
		Door door = new Door(m_imgDoorClosed, m_imgDoorOpened, 10, m_component, m_imob);
		door.setOpeningFrame(0, m_imgDoorOpening1, 120);
		door.setOpeningFrame(1, m_imgDoorOpening2, 120);
		door.setOpeningFrame(2, m_imgDoorOpening3, 120);
		door.setOpeningFrame(3, m_imgDoorOpening4, 120);

		ShootArea a = new ShootArea(door, 5, m_component, m_imob);
		
		a.setStateSleeping();
		
		return a;
	}
	
	/**
	 * CreateWindow
	 */
	private ShootArea CreateWindow()
	{
		Door door = new Door(m_imgWindowClosed, m_imgWindowOpened, 10,
						  m_component, m_imob);
		door.setOpeningFrame(0, m_imgWindowOpening1, 110);
		door.setOpeningFrame(1, m_imgWindowOpening2, 110);
		door.setOpeningFrame(2, m_imgWindowOpening3, 110);

		ShootArea a = new ShootArea(door, 5, m_component, m_imob);
		
		a.setStateSleeping();
		
		return a;
	}
	
	
	public void prepareForNewGame()
	{
		m_iLevel = 1;
		m_iScore = 0;
		m_iLives = INITIAL_NUMBER_OF_LIVES-2;
		repaintLives();
		repaintScores();
		repaintLevels();
		
		prepareForLevel(m_iLevel);
		setStatePlaying();
	}
	
	/**
	 * Prepares this Hotel Game for specified level:
	 *	- clears current possible guys from Doors, and puts new ones..
	 */
	public void prepareForLevel(int iLevel)
	{
		m_bLastLifeVisible = false;
		m_bLevelsVisible = true;
		m_bScoresVisible = true;
		
		setStatePlaying();
		m_iResponse = RESPONSE_NONE;

		m_iLevel = iLevel;
		
		repaintLives();
		repaintLevels();
		repaintScores();

		
		long lBadGuyMinStandingDelay = 500;
		long lBadGuyMaxStandingDelay = 2000;
		
		long lGoodGuyMinStandingDelay = 500;
		long lGoodGuyMaxStandingDelay = 2000;
		
		long lShootAreaMinOpeningDelay = 1000;
		long lShootAreaMaxOpeningDelay = 5000;

		long lBadGuyTriggeringFrame = 300;

		
		long lWndBadGuyMinStandingDelay = 500;
		long lWndBadGuyMaxStandingDelay = 2000;
		
		long lWndGoodGuyMinStandingDelay = 500;
		long lWndGoodGuyMaxStandingDelay = 2000;
		
		long lWndShootAreaMinOpeningDelay = 1000;
		long lWndShootAreaMaxOpeningDelay = 5000;
		
		long lWndBadGuyTriggeringFrame = 300;
		
		boolean bAddFaker = false;
		boolean bWndAddFaker = false;
		
		int level20 = m_iLevel;
		if (level20 > 20)
			level20 = 20;

		int level40 = m_iLevel;
		if (level40 > 40)
			level40 = 40;

		lBadGuyMinStandingDelay = 2000 - (1500 * level20 / 20);
		lBadGuyMaxStandingDelay = 6000 - (4000 * level20 / 20);
		
		lGoodGuyMinStandingDelay = 1000 - (500 * level20 / 20);
		lGoodGuyMaxStandingDelay = 7000 - (5000 * level20 / 20);
		
		lShootAreaMinOpeningDelay = 2000 - (1000 * level20 / 20);
		lShootAreaMaxOpeningDelay = 9000 - (7000 * level20 / 20);

		lBadGuyTriggeringFrame = 700 - (600 * level40 / 40);

		
		lWndBadGuyMinStandingDelay = 1500 - (1000 * level20 / 20);
		lWndBadGuyMaxStandingDelay = 7000 - (5500 * level20 / 20);
		
		lWndGoodGuyMinStandingDelay = 1000;
		lWndGoodGuyMaxStandingDelay = 5000;
		
		lWndShootAreaMinOpeningDelay = 5000;
		lWndShootAreaMaxOpeningDelay = 30000 - (20000 * level40/40);
		
		lWndBadGuyTriggeringFrame = 700 - (550 * level40 / 40);

		if (m_iLevel < 15)
			bAddFaker = false;
		else
			bAddFaker = true;
		
		if (m_iLevel < 30)
			bWndAddFaker = false;
		else
			bWndAddFaker = true;
		
		
		//
		// Prepare doors and windows
		//
		
		int i = 0;
		for (i=0; i<m_Doors.size(); i++)
		{
			ShootArea sa = (ShootArea) m_Doors.elementAt(i);
		
			// clear current contents..
			sa.removeAllGuys();
			
			// add bad guy
			
			BadGuy	guy = new BadGuy(m_imgBadGuyStanding, m_imgBadGuyDead, m_imgBadGuyDead, 4, 4, 5, m_component, m_imob);
			guy.setDyingFrame(0, m_imgBadGuyDying1, 0, 0, 100);
			guy.setDyingFrame(1, m_imgBadGuyDying2, 0, 0, 100);
			guy.setDyingFrame(2, m_imgBadGuyDying3, 0, 0, 100);
			guy.setDyingFrame(3, m_imgBadGuyDying4, 0, 0, 100);
			guy.setTriggeringFrame(0, m_imgBadGuyTriggering1, 0, 0, lBadGuyTriggeringFrame);
			guy.setTriggeringFrame(1, m_imgBadGuyTriggering2, 0, 0, lBadGuyTriggeringFrame);
			guy.setTriggeringFrame(2, m_imgBadGuyTriggering3, 0, 0, lBadGuyTriggeringFrame);
			guy.setTriggeringFrame(3, m_imgBadGuyTriggering4, 0, 0, lBadGuyTriggeringFrame);
			guy.setShootingFrame(0, m_imgBadGuyShooting4, 0, 0, 100);
			guy.setShootingFrame(1, m_imgBadGuyShooting3, 0, 0, 100);
			guy.setShootingFrame(2, m_imgBadGuyShooting2, 0, 0, 100);
			guy.setShootingFrame(3, m_imgBadGuyShooting1, 0, 0, 100);
			guy.setShootingFrame(4, m_imgBadGuyTriggering4, 0, 0, 100);
			guy.m_lMinStandingDelay = lBadGuyMinStandingDelay;
			guy.m_lMaxStandingDelay = lBadGuyMaxStandingDelay;
			
			int guyX = (sa.width - m_imgBadGuyStanding.getWidth(m_imob))/2;
			int guyY = sa.height - m_imgBadGuyStanding.getHeight(m_imob);
			guy.moveObject(guyX, guyY);
			guy.setHat(m_imgHat, guyX + 7, guyY - 27, m_imgHatDown, m_imgHatDownScore);
	
			sa.addBadGuy(guy, 100);
			
			// add good guy
			
			GoodGuy	guy2 = new GoodGuy(m_imgGoodGuyStanding, m_imgGoodGuyDead, 4, m_component, m_imob);
			guy2.setDyingFrame(0, m_imgBadGuyDying1, 0, 0, 100);
			guy2.setDyingFrame(1, m_imgBadGuyDying2, 0, 0, 100);
			guy2.setDyingFrame(2, m_imgBadGuyDying3, 0, 0, 100);
			guy2.setDyingFrame(3, m_imgBadGuyDying4, 0, 0, 100);
			guy2.m_lMinStandingDelay = lGoodGuyMinStandingDelay;
			guy2.m_lMaxStandingDelay = lGoodGuyMaxStandingDelay;
			
			int guy2X = (sa.width - m_imgBadGuyStanding.getWidth(m_imob))/2;
			int guy2Y = sa.height - m_imgBadGuyStanding.getHeight(m_imob);
			guy2.moveObject(guy2X, guy2Y);
			guy2.setHat(m_imgHat, guy2X + 7, guy2Y - 27, m_imgHatDown, m_imgHatDownScore);
	
			sa.addGoodGuy(guy2, 100);
			
			// and one more good guy
			
			GoodGuy	guy3 = new GoodGuy(m_imgGoodBadGuyStanding, m_imgGoodGuyDead, 4, m_component, m_imob);
			guy3.setDyingFrame(0, m_imgBadGuyDying1, 0, 0, 100);
			guy3.setDyingFrame(1, m_imgBadGuyDying2, 0, 0, 100);
			guy3.setDyingFrame(2, m_imgBadGuyDying3, 0, 0, 100);
			guy3.setDyingFrame(3, m_imgBadGuyDying4, 0, 0, 100);
			guy3.m_lMinStandingDelay = lGoodGuyMinStandingDelay;
			guy3.m_lMaxStandingDelay = lGoodGuyMaxStandingDelay;
			
			int guy3X = (sa.width - m_imgBadGuyStanding.getWidth(m_imob))/2;
			int guy3Y = sa.height - m_imgBadGuyStanding.getHeight(m_imob);
			guy3.moveObject(guy3X, guy3Y);
			guy3.setHat(m_imgHat, guy3X + 7, guy3Y - 27, m_imgHatDown, m_imgHatDownScore);
	
			sa.addGoodGuy(guy3, 100);

			if (bAddFaker)
			{
				// add bad guy
			
				BadGuy	guy4 = new BadGuy(m_imgGoodBadGuyStanding, m_imgBadGuyDead, m_imgBadGuyDead, 4, 4, 5, m_component, m_imob);
				guy4.setDyingFrame(0, m_imgBadGuyDying1, 0, 0, 100);
				guy4.setDyingFrame(1, m_imgBadGuyDying2, 0, 0, 100);
				guy4.setDyingFrame(2, m_imgBadGuyDying3, 0, 0, 100);
				guy4.setDyingFrame(3, m_imgBadGuyDying4, 0, 0, 100);
				guy4.setTriggeringFrame(0, m_imgBadGuyTriggering1, 0, 0, lBadGuyTriggeringFrame + 100);
				guy4.setTriggeringFrame(1, m_imgBadGuyTriggering2, 0, 0, lBadGuyTriggeringFrame);
				guy4.setTriggeringFrame(2, m_imgBadGuyTriggering3, 0, 0, lBadGuyTriggeringFrame);
				guy4.setTriggeringFrame(3, m_imgBadGuyTriggering4, 0, 0, lBadGuyTriggeringFrame);
				guy4.setShootingFrame(0, m_imgBadGuyShooting4, 0, 0, 100);
				guy4.setShootingFrame(1, m_imgBadGuyShooting3, 0, 0, 100);
				guy4.setShootingFrame(2, m_imgBadGuyShooting2, 0, 0, 100);
				guy4.setShootingFrame(3, m_imgBadGuyShooting1, 0, 0, 100);
				guy4.setShootingFrame(4, m_imgBadGuyTriggering4, 0, 0, 100);
				guy4.m_lMinStandingDelay = lBadGuyMinStandingDelay;
				guy4.m_lMaxStandingDelay = lBadGuyMaxStandingDelay;
			
				guyX = (sa.width - m_imgBadGuyStanding.getWidth(m_imob))/2;
				guyY = sa.height - m_imgBadGuyStanding.getHeight(m_imob);
				guy4.moveObject(guyX, guyY);
				guy4.setHat(m_imgHat, guyX + 7, guyY - 27, m_imgHatDown, m_imgHatDownScore);
	
				sa.addBadGuy(guy4, 100);
			}
			
			sa.m_lMinOpeningDelay = lShootAreaMinOpeningDelay;
			sa.m_lMaxOpeningDelay = lShootAreaMaxOpeningDelay;
			
			sa.setStateSleeping();
		}

		
		i = 0;
		for (i=0; i<m_Windows.size(); i++)
		{
			ShootArea sa = (ShootArea) m_Windows.elementAt(i);

			// clear current contents..
			sa.removeAllGuys();
			
			// add bad guy
			
			BadGuy	guy = new BadGuy(m_imgWndBadGuyStanding, m_imgWndBadGuyDead, m_imgWndBadGuyDead, 
									 4, 4, 5, m_component, m_imob);
			guy.setDyingFrame(0, m_imgWndGuyDying1, 0, 0, 100);
			guy.setDyingFrame(1, m_imgWndGuyDying2, 0, 0, 100);
			guy.setDyingFrame(2, m_imgWndGuyDying3, 0, 0, 100);
			guy.setDyingFrame(3, m_imgWndGuyDying4, 0, 0, 100);
			guy.setTriggeringFrame(0, m_imgWndBadGuyTriggering1, 0, 0, lWndBadGuyTriggeringFrame);
			guy.setTriggeringFrame(1, m_imgWndBadGuyTriggering2, 0, 0, lWndBadGuyTriggeringFrame);
			guy.setTriggeringFrame(2, m_imgWndBadGuyTriggering3, 0, 0, lWndBadGuyTriggeringFrame);
			guy.setTriggeringFrame(3, m_imgWndBadGuyTriggering4, 0, 0, lWndBadGuyTriggeringFrame);
			guy.setShootingFrame(0, m_imgWndBadGuyShooting1, 0, 0, 100);
			guy.setShootingFrame(1, m_imgWndBadGuyShooting2, 0, 0, 100);
			guy.setShootingFrame(2, m_imgWndBadGuyShooting3, 0, 0, 100);
			guy.setShootingFrame(3, m_imgWndBadGuyShooting4, 0, 0, 100);
			guy.setShootingFrame(4, m_imgWndBadGuyTriggering4, 0, 0, 100);
			guy.m_lMinStandingDelay = lWndBadGuyMinStandingDelay;
			guy.m_lMaxStandingDelay = lWndBadGuyMaxStandingDelay;
			
			int guyX = (sa.width - m_imgWndBadGuyStanding.getWidth(m_imob))/2;
			int guyY = sa.height - m_imgWndBadGuyStanding.getHeight(m_imob);
			guy.moveObject(guyX, guyY);
	
			sa.addBadGuy(guy, 100);

			// add good guy
			
			GoodGuy	guy2 = new GoodGuy(m_imgWndGoodGuyStanding, m_imgWndGoodGuyDead, 4, m_component, m_imob);
			guy2.setDyingFrame(0, m_imgWndGuyDying1, 0, 0, 100);
			guy2.setDyingFrame(1, m_imgWndGuyDying2, 0, 0, 100);
			guy2.setDyingFrame(2, m_imgWndGuyDying3, 0, 0, 100);
			guy2.setDyingFrame(3, m_imgWndGuyDying4, 0, 0, 100);
			guy2.m_lMinStandingDelay = lWndGoodGuyMinStandingDelay;
			guy2.m_lMaxStandingDelay = lWndGoodGuyMaxStandingDelay;
			
			int guy2X = (sa.width - m_imgWndGoodGuyStanding.getWidth(m_imob))/2;
			int guy2Y = sa.height - m_imgWndGoodGuyStanding.getHeight(m_imob);
			guy2.moveObject(guy2X, guy2Y);
	
			sa.addGoodGuy(guy2, 100);

			// and one more good guy
			
			GoodGuy	guy3 = new GoodGuy(m_imgWndGoodBadGuyStanding, m_imgWndGoodGuyDead, 4, m_component, m_imob);
			guy3.setDyingFrame(0, m_imgWndGuyDying1, 0, 0, 100);
			guy3.setDyingFrame(1, m_imgWndGuyDying2, 0, 0, 100);
			guy3.setDyingFrame(2, m_imgWndGuyDying3, 0, 0, 100);
			guy3.setDyingFrame(3, m_imgWndGuyDying4, 0, 0, 100);
			guy3.m_lMinStandingDelay = lWndGoodGuyMinStandingDelay;
			guy3.m_lMaxStandingDelay = lWndGoodGuyMaxStandingDelay;
			
			int guy3X = (sa.width - m_imgWndGoodBadGuyStanding.getWidth(m_imob))/2;
			int guy3Y = sa.height - m_imgWndGoodBadGuyStanding.getHeight(m_imob);
			guy3.moveObject(guy3X, guy3Y);
	
			sa.addGoodGuy(guy3, 100);

			if (bWndAddFaker)
			{
				BadGuy	guy4 = new BadGuy(m_imgWndGoodBadGuyStanding, m_imgWndBadGuyDead, m_imgWndBadGuyDead, 
										 4, 4, 5, m_component, m_imob);
				guy4.setDyingFrame(0, m_imgWndGuyDying1, 0, 0, 100);
				guy4.setDyingFrame(1, m_imgWndGuyDying2, 0, 0, 100);
				guy4.setDyingFrame(2, m_imgWndGuyDying3, 0, 0, 100);
				guy4.setDyingFrame(3, m_imgWndGuyDying4, 0, 0, 100);
				guy4.setTriggeringFrame(0, m_imgWndBadGuyTriggering1, 0, 0, lWndBadGuyTriggeringFrame + 100);
				guy4.setTriggeringFrame(1, m_imgWndBadGuyTriggering2, 0, 0, lWndBadGuyTriggeringFrame);
				guy4.setTriggeringFrame(2, m_imgWndBadGuyTriggering3, 0, 0, lWndBadGuyTriggeringFrame);
				guy4.setTriggeringFrame(3, m_imgWndBadGuyTriggering4, 0, 0, lWndBadGuyTriggeringFrame);
				guy4.setShootingFrame(0, m_imgWndBadGuyShooting1, 0, 0, 100);
				guy4.setShootingFrame(1, m_imgWndBadGuyShooting2, 0, 0, 100);
				guy4.setShootingFrame(2, m_imgWndBadGuyShooting3, 0, 0, 100);
				guy4.setShootingFrame(3, m_imgWndBadGuyShooting4, 0, 0, 100);
				guy4.setShootingFrame(4, m_imgWndBadGuyTriggering4, 0, 0, 100);
				guy4.m_lMinStandingDelay = lWndBadGuyMinStandingDelay;
				guy4.m_lMaxStandingDelay = lWndBadGuyMaxStandingDelay;
			
				guyX = (sa.width - m_imgWndBadGuyStanding.getWidth(m_imob))/2;
				guyY = sa.height - m_imgWndBadGuyStanding.getHeight(m_imob);
				guy4.moveObject(guyX, guyY);
	
				sa.addBadGuy(guy4, 100);

			}
			
			sa.m_lMinOpeningDelay = lWndShootAreaMinOpeningDelay;
			sa.m_lMaxOpeningDelay = lWndShootAreaMaxOpeningDelay;
			
			sa.setStateSleeping();
		}

	}

	/** PAINT HANDLERS *****************************************************/
	
	/**
	 * Paints entire content to destination
	 */
	public void paint(Graphics gDest, int iX, int iY)
	{
		super.paint(gDest, iX, iY);
	}
	
	/**
	 * Updates only changed content to destination
	 */
	public void update(Graphics gDest, int iX, int iY)
	{
		runScore();
		super.update(gDest, iX, iY);
	}
	
	/** ACTION *************************************************************/
	
	public void setStatePlaying()
	{
		m_iState = STATE_PLAYING;
		m_bLastLifeVisible = false;
		m_bLevelsVisible = true;
		m_bScoresVisible = true;
		
	}
	
	public void setStateReadyForLifeLess()
	{
		m_iState = STATE_READY_FOR_LIFE_LESS;
		m_iResponse = RESPONSE_READY_FOR_LIFE_LESS;
	}
	
	public void setStateReadyForGameOver()
	{
		m_iState = STATE_READY_FOR_GAME_OVER;
		m_iResponse = RESPONSE_READY_FOR_GAME_OVER;
	}
	
	public void setStateReadyForNextLevel()
	{
		m_iState = STATE_READY_FOR_NEXT_LEVEL;
		m_iResponse = RESPONSE_READY_FOR_NEXT_LEVEL;
	}
	
	/**
	 * Main action routine.
	 */
	public long doContinuousAction(long lTimeForAction)
	{
		// spend all given time..
		while(true)
		{
			if (m_iState == STATE_PLAYING)
				lTimeForAction = doStatePlaying(lTimeForAction);
			else if (m_iState == STATE_READY_FOR_LIFE_LESS)
				lTimeForAction = doStateReadyForLifeLess(lTimeForAction);
			else if (m_iState == STATE_READY_FOR_GAME_OVER)
				lTimeForAction = doStateReadyForGameOver(lTimeForAction);
			else if (m_iState == STATE_READY_FOR_NEXT_LEVEL)
				lTimeForAction = doStateReadyForNextLevel(lTimeForAction);
			else
				lTimeForAction = 0;
			
			if (lTimeForAction == 0)
				break;
		}
		return lTimeForAction;
	}
	
	private long doStatePlaying(long lTimeForAction)
	{
		int i;
	
		// if player shooted...
		if (m_bMouseClicked)
		{
			m_bMouseClicked = false;
		
			m_bSndPlayerGun = true;
			
			// Parse mouse click to areas
			for (i=0; i<m_Doors.size(); i++)
			{
				ShootArea sa = (ShootArea) m_Doors.elementAt(i);
				sa.onMouseClick(m_iMouseX, m_iMouseY);
			}
					
			for (i=0; i<m_Windows.size(); i++)
			{
				ShootArea sa = (ShootArea) m_Windows.elementAt(i);
				sa.onMouseClick(m_iMouseX, m_iMouseY);
			}
		}
		
		// Check responses
		for (i=0; i<m_Doors.size(); i++)
		{
			ShootArea sa = (ShootArea) m_Doors.elementAt(i);
			checkShootAreaResponse(sa, true);
		}
					
		for (i=0; i<m_Windows.size(); i++)
		{
			ShootArea sa = (ShootArea) m_Windows.elementAt(i);
			checkShootAreaResponse(sa, false);
		}
					
		checkForNextLevel();
		
		// return that all time was spent
		return 0;
	}

	private long doStateReadyForLifeLess(long lTimeForAction)
	{
		m_toggler.doAction(lTimeForAction);
		if (m_toggler.m_bChanged)
		{
			m_bLastLifeVisible = m_toggler.m_bFlag;
			repaintLives();
		}
		lTimeForAction = 0;
		return lTimeForAction;
	}

	private long doStateReadyForGameOver(long lTimeForAction)
	{
		return 0;
	}

	private long doStateReadyForNextLevel(long lTimeForAction)
	{
		// blink level numbers...
		m_toggler.doAction(lTimeForAction);
		if (m_toggler.m_bChanged)
		{
			m_bLevelsVisible = m_toggler.m_bFlag;
			repaintLevels();
		}
		
		lTimeForAction = 0;
		return lTimeForAction;
	}

	/** UTILITY CODE *******************************************************/
	
	private void checkShootAreaResponse(ShootArea sa, boolean bDoor)
	{
		if (sa.m_iResponse == sa.RESPONSE_PLAYER_SHOOT_HAT)
		{
			sa.m_iResponse = sa.RESPONSE_NONE;	// turn off - we should process it once
			m_iScore += SCORE_FOR_HAT;
			repaintScores();
			m_bSndPlayerGunHat = true;
			m_bSndPlayerGun = false;
		}
		else if (sa.m_iResponse == sa.RESPONSE_PLAYER_SHOOT_BAD_GUY)
		{
			sa.m_iResponse = sa.RESPONSE_NONE;	// turn off - we should process it once
			if (bDoor)
				m_iScore += SCORE_FOR_BAD_GUY_IN_DOOR;
			else
				m_iScore += SCORE_FOR_BAD_GUY_IN_WINDOW;
			repaintScores();
			m_bSndPlayerGun = true;
		}
		else if ((sa.m_iResponse == sa.RESPONSE_PLAYER_SHOOT_GOOD_GUY) ||
				(sa.m_iResponse == sa.RESPONSE_BAD_GUY_SHOOT_PLAYER))
		{
			m_bSndPlayerGun = true;
			sa.m_iResponse = sa.RESPONSE_NONE;	// turn off - we should process it once
			if (m_iLives == 0)
			{
				setStateReadyForGameOver();
			}
			else
			{
				m_iLives--;
				setStateReadyForLifeLess();
			}
		}
	}
	
	private void checkForNextLevel()
	{
		// check if player has enough score for next level..
		if (m_iScore >= m_iLevel * (SCORES_PER_LEVEL + m_iLevel * SCORES_ADD_PER_LEVEL))
		{
			m_iRunningScore = m_iScore;
			repaintScores();
			setStateReadyForNextLevel();
		}
	}

	/**
	 * Paint life icons..
	 * When m_iLives is e.g. 2 - icon 0 and 1 will be painted
	 * and icon 2 may blink if STATE_READY_FOR_LIFE_LESS (user has just lost life)
	 */
	protected void repaintLives()
	{
		for (int i = 0; i < MAX_LIVES; i++)
		{
			if (i < m_iLives)
				showObject(m_bbLives[i]);
			else
				hideObject(m_bbLives[i]);
		}

		// check to blink last life
		if (m_iLives < MAX_LIVES)
		{
			if (m_iState == STATE_READY_FOR_LIFE_LESS)
			{
				if (m_bLastLifeVisible)
					showObject(m_bbLives[m_iLives]);
				else
					hideObject(m_bbLives[m_iLives]);
			}
			else
				hideObject(m_bbLives[m_iLives]);
		}
	}
	
	protected void repaintLevels()
	{
		String s = "" + m_iLevel;
		s = lpad(s, MAX_LEVEL_DIGITS, '0');

		for ( int i=0; i<MAX_LEVEL_DIGITS; i++ )
		{
			int iIndex = s.charAt(i) - '0';
			if ( (iIndex >= 0) && (iIndex <= 9) )
			{
				if (iIndex == 0)
					m_bbLevelDigits[i].setImage(m_imgNumber0);
				else if (iIndex == 1)
					m_bbLevelDigits[i].setImage(m_imgNumber1);
				else if (iIndex == 2)
					m_bbLevelDigits[i].setImage(m_imgNumber2);
				else if (iIndex == 3)
					m_bbLevelDigits[i].setImage(m_imgNumber3);
				else if (iIndex == 4)
					m_bbLevelDigits[i].setImage(m_imgNumber4);
				else if (iIndex == 5)
					m_bbLevelDigits[i].setImage(m_imgNumber5);
				else if (iIndex == 6)
					m_bbLevelDigits[i].setImage(m_imgNumber6);
				else if (iIndex == 7)
					m_bbLevelDigits[i].setImage(m_imgNumber7);
				else if (iIndex == 8)
					m_bbLevelDigits[i].setImage(m_imgNumber8);
				else if (iIndex == 9)
					m_bbLevelDigits[i].setImage(m_imgNumber9);
				else
					m_bbLevelDigits[i].setImage(m_imgNumberNull);
			}
		}
	}

	/** SCORE **************************************************************/
	
	int m_iRunningScore = 0;

	private void runScore()
	{
		if (m_iRunningScore < m_iScore)
		{
			m_iRunningScore += 3;
			if (m_iRunningScore > m_iScore)
				m_iRunningScore = m_iScore;
			repaintScores();
		}
	}
	
	
	protected void repaintScores()
	{
		if (m_iScore < m_iRunningScore)
			m_iRunningScore = m_iScore; 
		
		String s = "" + m_iRunningScore;
		s = lpad(s, MAX_SCORE_DIGITS, '0');

		for ( int i=0; i<MAX_SCORE_DIGITS; i++ )
		{
			int iIndex = s.charAt(i) - '0';
			if ( (iIndex >= 0) && (iIndex <= 9) )
			{
				if (iIndex == 0)
					m_bbScoreDigits[i].setImage(m_imgNumber0);
				else if (iIndex == 1)
					m_bbScoreDigits[i].setImage(m_imgNumber1);
				else if (iIndex == 2)
					m_bbScoreDigits[i].setImage(m_imgNumber2);
				else if (iIndex == 3)
					m_bbScoreDigits[i].setImage(m_imgNumber3);
				else if (iIndex == 4)
					m_bbScoreDigits[i].setImage(m_imgNumber4);
				else if (iIndex == 5)
					m_bbScoreDigits[i].setImage(m_imgNumber5);
				else if (iIndex == 6)
					m_bbScoreDigits[i].setImage(m_imgNumber6);
				else if (iIndex == 7)
					m_bbScoreDigits[i].setImage(m_imgNumber7);
				else if (iIndex == 8)
					m_bbScoreDigits[i].setImage(m_imgNumber8);
				else if (iIndex == 9)
					m_bbScoreDigits[i].setImage(m_imgNumber9);
				else
					m_bbScoreDigits[i].setImage(m_imgNumberNull);
				
				if (m_bScoresVisible)
					showObject(m_bbScoreDigits[i]);
				else
					hideObject(m_bbScoreDigits[i]);
			}
		}
	}

	/**
	 * public String lpad(String str, int places, char fill_char)
	 */
	private String lpad(String str, int places, char fill_char)
	{
		int l = str.length();
		while (l < places)
		{
			str = fill_char + str;
			l++;
		}
		return str;
	}	


	/** MOUSE HANDLERS *****************************************************/

	// mouse indicators that are currently processed
	public int		m_iMouseX;
	public int		m_iMouseY;
	public boolean	m_bMouseClicked = false;		// true when clicked
}
