// TerrainMap.cpp: implementation of the TerrainMap class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "TerrainMap.h"
#include <tomaskovic\util\MemoryManager.h>
#include <tomaskovic\util\BitmapRGB.h>


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

TerrainMap::TerrainMap() {
	iWidth = 0;
	iHeight = 0;
	pfMap = NULL;
	pNormals = NULL;
}


TerrainMap::~TerrainMap() {
	MemFree(pfMap);
}


void TerrainMap::Create(int width, int height) {

	iWidth = width;
	iHeight = height;
	pfMap = (float*) MemAlloc(sizeof(float) * iWidth * iHeight);
	for (int y=0; y<iHeight; y++) {
		for (int x=0; x<iWidth; x++) {
			pfMap[x+y*iHeight] = 1.0f + x/(y+1);
		}
	}

	CalculateNormals();
}

BOOL TerrainMap::LoadFromBMP(char* pszPathFilename) {

	BitmapRGB bmp;
	if (!bmp.Load(pszPathFilename))
		return FALSE;

	iWidth = bmp.iWidth;
	iHeight = bmp.iHeight;
	pfMap = (float*) MemAlloc(sizeof(float) * iWidth * iHeight);
	for (int y=0; y<iHeight; y++) {
		for (int x=0; x<iWidth; x++) {
			float fAlt = (float) bmp.pBits[x*3 + y*bmp.iBytesPerLine];
			pfMap[x+y*iHeight] = fAlt / 16;
		}
	}

	CalculateNormals();
	return TRUE;
}

void TerrainMap::SetAltitude(float fAlt) {
	for (int y=0; y<iHeight; y++) {
		for (int x=0; x<iWidth; x++) {
			pfMap[x+y*iHeight] = pfMap[x+y*iHeight] * fAlt;
		}
	}

	CalculateNormals();
}

void TerrainMap::CalculateNormals() {

	// Calc normals
	pNormals = (Vector3f*) MemAlloc(sizeof(Vector3f) * iWidth * iHeight);

	/*
	for (int y=0; y<iHeight-1; y++) {
		for (int x=0; x<iWidth-1; x++) {

			float fX = (float) x;
			float fZ = (float) y;

			// 1 2
			// 3 

			float f1 = pfMap[x+y*iHeight];
			float f2 = pfMap[x+1+y*iHeight];
			float f3 = pfMap[x+(y+1)*iHeight];

			Vector3f v1, v2, v3;
			v1.Set(fX, f1, fZ);
			v2.Set(fX+1, f2, fZ);
			v3.Set(fX, f3, fZ+1);

			Vector3f v21, v31;
			v21.Set(v2);
			v21.Substract(v1);
			v31.Set(v3);
			v31.Substract(v1);

			v21.CrossProduct(v31);
			v21.Normalize();

			memcpy(&pNormals[x + y*iHeight], &v21, sizeof(Vector3f));
		}
	}
	*/


	for (int y=1; y<iHeight-1; y++) {
		for (int x=1; x<iWidth-1; x++) {

			float fX = (float) x;
			float fZ = (float) y;

			// we're on point 5:
			// 1 2-3
			//  /|/|
			// 4-5-6
			// |/|/ 
			// 7-8 9

			Vector3f v2, v3, v4, v5, v6, v7, v8;

			v2.Set(fX, pfMap[x+(y-1)*iHeight], fZ-1);
			v3.Set(fX+1, pfMap[x+1+(y-1)*iHeight], fZ-1);

			v4.Set(fX-1, pfMap[x-1+y*iHeight], fZ);
			v5.Set(fX, pfMap[x+y*iHeight], fZ);
			v6.Set(fX+1, pfMap[x+1+y*iHeight], fZ);

			v7.Set(fX-1, pfMap[x-1+(y+1)*iHeight], fZ+1);
			v8.Set(fX, pfMap[x+(y+1)*iHeight], fZ+1);

			Vector3f v54, v52, v53, v56, v58, v57;
			v54.Set(v4); v54.Substract(v5);
			v52.Set(v2); v52.Substract(v5);
			v53.Set(v3); v53.Substract(v5);
			v56.Set(v6); v56.Substract(v5);
			v58.Set(v8); v58.Substract(v5);
			v57.Set(v7); v57.Substract(v5);

			Vector3f v254, v352, v653, v856, v758, v457;
			v254.CrossProduct(v52, v54);
			v352.CrossProduct(v53, v52);
			v653.CrossProduct(v56, v53);
			v856.CrossProduct(v58, v56);
			v758.CrossProduct(v57, v58);
			v457.CrossProduct(v54, v57);

			Vector3f vAll;
			vAll.Set(v254);
			vAll.Add(v352);
			vAll.Add(v653);
			vAll.Add(v856);
			vAll.Add(v758);
			vAll.Add(v457);
			vAll.Normalize();

			vAll.Normalize();
			vAll.x *= -1;
			vAll.y *= -1;
			vAll.z *= -1;

			memcpy(&pNormals[x + y*iHeight], &vAll, sizeof(Vector3f));
		}
	}

}

void TerrainMap::Draw(GLTextureManager &glTextureManager) {

	int t = glTextureManager.FindGLTextureID(iTextureID);

	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, t);
	glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

	glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

    // Paint every triangle face
    glBegin(GL_TRIANGLES); // GL_TRIANGLES

	glColor3f(1.0f, 1.0f, 1.0f);

	for (int y=0; y<iHeight-1; y++) {
		for (int x=0; x<iWidth-1; x++) {

			float fX = (float) x;
			float fZ = (float) y;

			// 1 2
			// 3 4
			float f1 = pfMap[x+y*iHeight];
			float f2 = pfMap[x+1+y*iHeight];
			float f3 = pfMap[x+(y+1)*iHeight];
			float f4 = pfMap[x+1+(y+1)*iHeight];

			Vector3f &n1 = pNormals[x + y*iHeight];
			Vector3f &n2 = pNormals[x+1 + y*iHeight];
			Vector3f &n3 = pNormals[x + (y+1)*iHeight];
			Vector3f &n4 = pNormals[x+1 + (y+1)*iHeight];

			glTexCoord2f(fX / 256.0f, fZ / 256.0f);
			glNormal3f(n1.x, n1.y, n1.z);
			glVertex3f(fX, f1, fZ);

			glTexCoord2f((fX+1) / 256.0f, fZ / 256.0f);
			glNormal3f(n2.x, n2.y, n2.z);
			glVertex3f(fX+1, f2, fZ);

			glTexCoord2f(fX / 256.0f, (fZ+1) / 256.0f);
			glNormal3f(n3.x, n3.y, n3.z);
			glVertex3f(fX, f3, fZ+1);


			glTexCoord2f(fX / 256.0f, (fZ+1) / 256.0f);
			glNormal3f(n3.x, n3.y, n3.z);
			glVertex3f(fX, f3, fZ+1);

			glTexCoord2f((fX+1) / 256.0f, fZ / 256.0f);
			glNormal3f(n2.x, n2.y, n2.z);
			glVertex3f(fX+1, f2, fZ);

			glTexCoord2f((fX+1) / 256.0f, fZ / 256.0f);
			glNormal3f(n4.x, n4.y, n4.z);
			glVertex3f(fX+1, f4, fZ+1);
		}
	}
    glEnd();

	glDisable(GL_TEXTURE_2D);

	/*
    // Paint every triangle face
    glBegin(GL_TRIANGLES); // GL_TRIANGLES

	glColor3f(1.0f, 1.0f, 1.0f);

	for (int y=0; y<iHeight-1; y++) {
		for (int x=0; x<iWidth-1; x++) {

			float fX = (float) x;
			float fZ = (float) y;

			// 1 2
			// 3 4
			float f1 = pfMap[x+y*iHeight];
			float f2 = pfMap[x+1+y*iHeight];
			float f3 = pfMap[x+(y+1)*iHeight];
			float f4 = pfMap[x+1+(y+1)*iHeight];

			Vector3f &n1 = pNormals[x + y*iHeight];
			Vector3f &n2 = pNormals[x+1 + y*iHeight];
			Vector3f &n3 = pNormals[x + (y+1)*iHeight];
			Vector3f &n4 = pNormals[x+1 + (y+1)*iHeight];

			glNormal3f(n1.x, n1.y, n1.z);
			glVertex3f(fX, f1, fZ);
			glNormal3f(n2.x, n2.y, n2.z);
			glVertex3f(fX+1, f2, fZ);
			glNormal3f(n3.x, n3.y, n3.z);
			glVertex3f(fX, f3, fZ+1);

			glNormal3f(n3.x, n3.y, n3.z);
			glVertex3f(fX, f3, fZ+1);
			glNormal3f(n2.x, n2.y, n2.z);
			glVertex3f(fX+1, f2, fZ);
			glNormal3f(n4.x, n4.y, n4.z);
			glVertex3f(fX+1, f4, fZ+1);
		}
	}
    glEnd();
	*/

	/*
    // Paint every triangle face
    glBegin(GL_TRIANGLE_STRIP);

	glColor3f(1.0f, 1.0f, 1.0f);

	for (int y=0; y<iHeight; y++) {
		for (int x=0; x<iWidth-1; x++) {

			float fX = (float) x;
			float fZ = (float) y;

			// 1 2
			float f1 = pfMap[x+y*iHeight];
			float f2 = pfMap[x+1+y*iHeight];

			Vector3f &n1 = pNormals[x + y*iHeight];
			Vector3f &n2 = pNormals[x+1 + y*iHeight];

			glNormal3f(n1.x, n1.y, n1.z);
			glVertex3f(fX, f1, fZ);
			glNormal3f(n2.x, n2.y, n2.z);
			glVertex3f(fX+1, f2, fZ);
		}
	}
    glEnd();
	*/

}