/**
 * Autodesk3dsDatabase.cpp
 *
 * @author Matija Tomaskovic
 * @version 21-Jul-2001
 */

#include "Autodesk3dsDatabase.h"
#include <tomaskovic\util\FileInputStream.h>
#include <tomaskovic\util\BufferInputStream.h>
#include <tomaskovic\util\SuperString.h>
#include <tomaskovic\util\MemoryManager.h>
#include "Vector3f.h"


Autodesk3dsDatabase::Autodesk3dsDatabase() {
}

Autodesk3dsDatabase::~Autodesk3dsDatabase() {
	// Delete all meshes..
	for (int i=0; i<vMeshes.Size(); i++) {
		A3DS_MESH* p = (A3DS_MESH*) vMeshes.ElementAt(i);
		MemFree(p->vertexArray);
		MemFree(p->faceArray);
		MemFree(p);
	}
}

BOOL Autodesk3dsDatabase::LoadFromFile(char* psz3dsFilename) {

	strPathFilename.Set(psz3dsFilename);

	FileInputStream fis;
	if (!fis.Open(psz3dsFilename)) {
		SetLastError("Failed to open 3ds file");
		return FALSE;
	}

	//
	// Read chunk
	//

	A3DS_CHUNK_HEADER headerMain;
	if (!LoadChunkHeader(fis, &headerMain))
		return FALSE;

	if (headerMain.wChunkID != A3DS_CHUNK_TYPE_MAIN) {
		SetLastError("Main chunk not found at beginning of the file");
		return FALSE;
	}

	// Read data of main chunk

	Buffer bufferMain;
	if (!LoadChunkData(fis, headerMain, &bufferMain))
		return FALSE;

	BufferInputStream bisMain(&bufferMain);

	//
	// Process main chunk
	//

	while(TRUE) {
		if (bisMain.IsEndOfStream())
			break;
		A3DS_CHUNK_HEADER header;
		if (!LoadChunkHeader(bisMain, &header))
			return FALSE;

		switch (header.wChunkID)
		{
			case A3DS_CHUNK_TYPE_EDITOR: 
				{
					Buffer editor;
					if (!LoadChunkData(bisMain, header, &editor)) return FALSE;
					BufferInputStream bis(&editor);
					if (!ProcessEditorChunk(bis)) return FALSE;
				}
				break;

			case A3DS_CHUNK_TYPE_KEYFRAMER_CONFIG: 
				{
				}
				break;

			default: 
				{
					// Skip chunk
					//::MessageBox(NULL, "Unrecognized data chunk in 3ds file - may not load correctly", "WARNING", MB_OK);
					if (!SkipChunk(bisMain, header)) return FALSE;
				}
				break;
		}
	}



	return TRUE;
}


BOOL Autodesk3dsDatabase::LoadChunkData(InputStream &isChunkData, A3DS_CHUNK_HEADER &chunkHeader, Buffer* pBufferDest) {

	int iLen = chunkHeader.dwChunkSize - sizeof(A3DS_CHUNK_HEADER);
	pBufferDest->SetSize(iLen);

	if (isChunkData.ReadBytes(pBufferDest->GetBuffer(), iLen) != iLen) {
		SetLastError("Failed to load data of chunk");
		return FALSE;
	}

	return TRUE;
}

BOOL Autodesk3dsDatabase::LoadChunkHeader(InputStream &isChunkData, A3DS_CHUNK_HEADER* pChunkHeaderDest) {
	if (isChunkData.ReadBytes((BYTE*) pChunkHeaderDest, sizeof(A3DS_CHUNK_HEADER)) != 
		sizeof(A3DS_CHUNK_HEADER)) 
	{
		SetLastError("Failed to load data of chunk");
		return FALSE;
	}

	return TRUE;
}

BOOL Autodesk3dsDatabase::SkipChunk(InputStream &is, A3DS_CHUNK_HEADER &header) {
	int iLen = header.dwChunkSize - sizeof(A3DS_CHUNK_HEADER);
	for (int i=0; i<iLen; i++) {
		if (is.Read() == -1) {
			SetLastError("Failed to load data to skip unknown chunk in Main");
			return FALSE;
		}
	}
	return TRUE;
}


BOOL Autodesk3dsDatabase::ProcessEditorChunk(InputStream &isChunkData) {
	while(TRUE) {
		if (isChunkData.IsEndOfStream())
			return TRUE;
		A3DS_CHUNK_HEADER header;
		if (!LoadChunkHeader(isChunkData, &header))
			return FALSE;

		switch (header.wChunkID)
		{
			case A3DS_CHUNK_TYPE_EDITOR_OBJECT_DESCRIPTION:
				{
					// Load object name (zero-terminated string)
					SuperString strName;
					while (TRUE) {
						int b = isChunkData.Read();
						if (b == -1) {
							SetLastError("Object name expected");
							return FALSE;
						}
						if (b == 0)
							break;
						strName.Append((char) b);
					}

					// Triangle list, light or camera follows..

					A3DS_CHUNK_HEADER header;
					if (!LoadChunkHeader(isChunkData, &header))
						return FALSE;

					switch (header.wChunkID)
					{
						case A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLES:
							{
								A3DS_MESH mesh;
								memset(&mesh, 0, sizeof(A3DS_MESH));

								if (strName.GetLength() < 11)
									strcpy(mesh.szName, strName.GetBuffer());
								else
									strcpy(mesh.szName, "Name to long");

								Buffer buffer;
								if (!LoadChunkData(isChunkData, header, &buffer)) return FALSE;
								BufferInputStream isChunkData(&buffer);

								while(TRUE) {
									if (isChunkData.IsEndOfStream())
										break;
									A3DS_CHUNK_HEADER header;
									if (!LoadChunkHeader(isChunkData, &header)) {
										return FALSE;
									}

									switch (header.wChunkID)
									{
										case A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_VERTEX_LIST:
											{
												if (mesh.vertexArray) {
													SetLastError("Vertex array already loaded");
													return FALSE;
												}

												if (isChunkData.ReadBytes((BYTE*) &mesh.wVertices, sizeof(WORD)) != sizeof(WORD))
													return FALSE;

												mesh.vertexArray = (A3DS_VERTEX*) MemAlloc(sizeof(A3DS_VERTEX) * mesh.wVertices);
												for (int i=0; i<mesh.wVertices; i++) {
													if (isChunkData.ReadBytes((BYTE*) &mesh.vertexArray[i], 
															sizeof(A3DS_VERTEX)) != sizeof(A3DS_VERTEX)) 
													{
														MemFree(mesh.vertexArray);
														return FALSE;
													}
												}
											}
											break;
										case A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_VERTEX_OPTIONS:
											if (!SkipChunk(isChunkData, header)) return FALSE;
											break;
										case A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_FACE_LIST:
											{
												if (mesh.faceArray) {
													SetLastError("Face array already loaded");
													return FALSE;
												}

												if (isChunkData.ReadBytes((BYTE*) &mesh.wFaces, sizeof(WORD)) != sizeof(WORD))
													return FALSE;

												mesh.faceArray = (A3DS_FACE*) MemAlloc(sizeof(A3DS_FACE) * mesh.wFaces);
												for (int i=0; i<mesh.wFaces; i++) {
													if (isChunkData.ReadBytes((BYTE*) &mesh.faceArray[i], 
															sizeof(A3DS_FACE)) != sizeof(A3DS_FACE)) 
													{
														MemFree(mesh.faceArray);
														return FALSE;
													}

													/*
													// Calc face normal
													Vector3f v1, v2, v3;
													A3DS_VERTEX *pV1 = (A3DS_VERTEX*) &mesh.vertexArray[mesh.faceArray[i].wVertexIndex1];
													A3DS_VERTEX *pV2 = (A3DS_VERTEX*) &mesh.vertexArray[mesh.faceArray[i].wVertexIndex2];
													A3DS_VERTEX *pV3 = (A3DS_VERTEX*) &mesh.vertexArray[mesh.faceArray[i].wVertexIndex3];
													v1.Set(pV1->x, pV1->y, pV1->z);
													v2.Set(pV2->x, pV2->y, pV2->z);
													v3.Set(pV3->x, pV3->y, pV3->z);
													v1.Substract(v2);
													v3.Substract(v3);
													v1.CrossProduct(v3);
													v1.Normalize();
													mesh.faceArray[i].faceNormal.x = v1.x;
													mesh.faceArray[i].faceNormal.y = v1.y;
													mesh.faceArray[i].faceNormal.z = v1.z;

													// invert face..
													WORD w = mesh.faceArray[i].wVertexIndex1;
													mesh.faceArray[i].wVertexIndex1 = mesh.faceArray[i].wVertexIndex2;
													mesh.faceArray[i].wVertexIndex2 = w;
													*/
												}
											}
											break;
										case A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_FACE_MATERIAL:
											if (!SkipChunk(isChunkData, header)) return FALSE;
											break;
										case A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_MAPPING_COORDINATES:
											if (!SkipChunk(isChunkData, header)) return FALSE;
											break;
										case A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_FACE_SMOOTHING_GROUP:
											if (!SkipChunk(isChunkData, header)) return FALSE;
											break;
										case A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_TRANSLATION_MATRIX:
											if (!SkipChunk(isChunkData, header)) return FALSE;
											break;
										case A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_OBJECT_VISIBLE:
											if (!SkipChunk(isChunkData, header)) return FALSE;
											break;
										case A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_STANDARD_MAPPING:
											if (!SkipChunk(isChunkData, header)) return FALSE;
											break;

										default: 
											if (!SkipChunk(isChunkData, header)) return FALSE;
											break;
									}
								}

								A3DS_MESH* pMesh = (A3DS_MESH*) MemAlloc(sizeof(A3DS_MESH));
								memcpy(pMesh, &mesh, sizeof(A3DS_MESH));
								vMeshes.AddElement(pMesh);
							}
							break;
						case A3DS_CHUNK_TYPE_EDITOR_OBJECT_LIGHT:
							if (!SkipChunk(isChunkData, header)) return FALSE;
							break;
						case A3DS_CHUNK_TYPE_EDITOR_OBJECT_CAMERA:
							if (!SkipChunk(isChunkData, header)) return FALSE;
							break;
						default: 
							if (!SkipChunk(isChunkData, header)) return FALSE;
							break;
					}
				}
				break;

			case A3DS_CHUNK_TYPE_EDITOR_MAIN_BLOCK:
			case A3DS_CHUNK_TYPE_EDITOR_PART_OF_CONFIG:
			case A3DS_CHUNK_TYPE_EDITOR_START_OF_VIEWPORT_INDICATOR:
			case A3DS_CHUNK_TYPE_EDITOR_VIEWPORT_DEFINITION_TYPE_2:
			case A3DS_CHUNK_TYPE_EDITOR_VIEWPORT_DEFINITION_TYPE_1:
			case A3DS_CHUNK_TYPE_EDITOR_VIEWPORT_DEFINITION_TYPE_3:
			default: 
				if (!SkipChunk(isChunkData, header)) return FALSE;
				break;
		}
	}

	return TRUE;
}


void Autodesk3dsDatabase::SetLastError(char *pszErrorText) {
	::MessageBox(NULL, pszErrorText, "ERROR", MB_OK);
}


A3DS_MESH* Autodesk3dsDatabase::GetMeshByName(char *pszMeshName) {
	for (int i=0; i<vMeshes.Size(); i++) {
		A3DS_MESH* p = (A3DS_MESH*) vMeshes.ElementAt(i);
		if (strcmp(pszMeshName, p->szName) == 0)
			return p;
	}
	return NULL;
}
