/**
 * Autodesk3dsDatabase.h
 *
 * Autodesk3dsDatabase class loads all meshes (objects) from Autodesk
 * 3DS files.
 *
 * @author Matija Tomaskovic
 * @version 21-Jul-2001
 */

#if !defined(_Autodesk3dsDatabase_H__INCLUDED_)
#define _Autodesk3dsDatabase_H__INCLUDED_

#include <windows.h>
#include "GLTextureManager.h"
#include <tomaskovic\util\Buffer.h>
#include <tomaskovic\util\InputStream.h>
#include <tomaskovic\util\Vector.h>
#include <tomaskovic\util\SuperString.h>


#pragma pack(push, a3ds)
#pragma pack(1)

    // 3DS file is built of chunks that consist of chunk header and chunk data.
    // Chunk data can consist of subchunks.

	typedef struct {
		WORD wChunkID;
		DWORD dwChunkSize;
	} A3DS_CHUNK_HEADER;

    // Vertex
	typedef struct {
		float x;
		float y;
		float z;
	} A3DS_VERTEX;

    // Face is defined with zero-based indexes of vertices
	typedef struct {
		WORD wVertexIndex1;
		WORD wVertexIndex2;
		WORD wVertexIndex3;
		WORD wFaceInfo;
	} A3DS_FACE;

    // A3DS_MESH is autodesk 3ds mesh, that is having array of vertices
    // and faces who indexes these vertices (zero-indexes)
	typedef struct {
        char szName[25];    // 3DS uses max. 11 chars?
		WORD wVertices;
        A3DS_VERTEX *vertexArray;   // array of A3DS_VERTEX (wVertices)
		WORD wFaces;
        A3DS_FACE *faceArray;       // array of A3DS_FACE (wFaces)
	} A3DS_MESH;

#pragma pack(pop, a3ds)

// Chunk types (A3DS_CHUNK_HEADER.wChunkID)
#define	A3DS_CHUNK_TYPE_MAIN				0x4d4d
#define A3DS_CHUNK_TYPE_EDITOR				0x3d3d
	#define A3DS_CHUNK_TYPE_EDITOR_MAIN_BLOCK		0x3d3e
	#define A3DS_CHUNK_TYPE_EDITOR_PART_OF_CONFIG	0x0100
	#define A3DS_CHUNK_TYPE_EDITOR_START_OF_VIEWPORT_INDICATOR	0x7001
		#define A3DS_CHUNK_TYPE_EDITOR_VIEWPORT_DEFINITION_TYPE_2	0x7011
		#define A3DS_CHUNK_TYPE_EDITOR_VIEWPORT_DEFINITION_TYPE_1	0x7012
		#define A3DS_CHUNK_TYPE_EDITOR_VIEWPORT_DEFINITION_TYPE_3	0x7020
	#define A3DS_CHUNK_TYPE_EDITOR_OBJECT_DESCRIPTION	0x4000
		#define A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLES  			0x4100
			#define A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_VERTEX_LIST		0x4110
			#define A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_VERTEX_OPTIONS	0x4111
			#define A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_FACE_LIST		0x4120
			#define A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_FACE_MATERIAL	0x4130
			#define A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_MAPPING_COORDINATES	0x4140
			#define A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_FACE_SMOOTHING_GROUP	0x4150
			#define A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_TRANSLATION_MATRIX	0x4160
			#define A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_OBJECT_VISIBLE		0x4165
			#define A3DS_CHUNK_TYPE_EDITOR_OBJECT_TRIANGLE_STANDARD_MAPPING		0x4170
		#define A3DS_CHUNK_TYPE_EDITOR_OBJECT_LIGHT      			0x4600
		#define A3DS_CHUNK_TYPE_EDITOR_OBJECT_CAMERA     			0x4700


#define A3DS_CHUNK_TYPE_KEYFRAMER_CONFIG	0xb000
//#define A3DS_CHUNK_TYPE_

class Autodesk3dsDatabase
{
public:

    Autodesk3dsDatabase();
    ~Autodesk3dsDatabase();

    BOOL LoadFromFile(char* psz3dsFilename);
	A3DS_MESH* GetMeshByName(char* pszMeshName);
	Vector vMeshes;	// Vector of A3DS_MESH structures

	SuperString strPathFilename;

private:

	BOOL SkipChunk(InputStream &is, A3DS_CHUNK_HEADER &header);
	BOOL ProcessEditorChunk(InputStream &is);
	BOOL LoadChunkHeader(InputStream &isChunkData, A3DS_CHUNK_HEADER* pChunkHeaderDest);
	BOOL LoadChunkData(InputStream &isChunkData, A3DS_CHUNK_HEADER &chunkHeader, Buffer* pBufferDest);

	void SetLastError(char* pszErrorText);

};

#endif // !defined(_Autodesk3dsDatabase_H__INCLUDED_)
