/**
 * Camera.cpp
 * Implementation of the Camera class.
 *
 * @author Matija Tomaskovic
 * @version 21-May-2001
 */

#include "Camera.h"
#include "Vector3f.h"
#include "Matrix44f.h"
#include "Quaternion.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Camera::Camera() {
    fFOV = 45.0f;

    iWindowWidth = 0;
    iWindowHeight = 0;

    fEyeX = 0.0f;
    fEyeY = 10.0f;
    fEyeZ = 20.0f;
    fTargetX = 0.0f;
    fTargetY = 0.0f;
    fTargetZ = 0.0f;
    fUpX = 0.0f;
    fUpY = 1.0f;
    fUpZ = 0.0f;

    fNear = 0.1f;
    fFar = 1000.0f;
}

Camera::~Camera() {
}

void Camera::SetViewportSize(int iWindowWidth, int iWindowHeight) {
	this->iWindowWidth = iWindowWidth;
	this->iWindowHeight = iWindowHeight;
}

void Camera::SetOrientation(float fEyeX, float fEyeY, float fEyeZ,
                float fTargetX, float fTargetY, float fTargetZ,
                float fUpX, float fUpY, float fUpZ)
{
    this->fEyeX = fEyeX;
    this->fEyeY = fEyeY;
    this->fEyeZ = fEyeZ;
    this->fTargetX = fTargetX;
    this->fTargetY = fTargetY;
    this->fTargetZ = fTargetZ;
    this->fUpX = fUpX;
    this->fUpY = fUpY;
    this->fUpZ = fUpZ;
}


void Camera::ApplyToGLViewport() {

    // Transform GL coordinate system according to camera

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
    gluPerspective(fFOV, (GLfloat) iWindowWidth/(GLfloat) iWindowHeight, fNear, fFar);

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

    gluLookAt( fEyeX, fEyeY, fEyeZ,
            fTargetX, fTargetY, fTargetZ,
            fUpX, fUpY, fUpZ);
}

void Camera::SetFieldOfView(float fFOV) {
	this->fFOV = fFOV;
}

void Camera::ApplyPitch(float fAngle) {
	//
	// Rotate Direction around (Up x Direction)
	//

	// Direction = Target - eye
	Vector3f vDirection, vEye;
	vDirection.Set(fTargetX, fTargetY, fTargetZ);
	vEye.Set(fEyeX, fEyeY, fEyeZ);
	vDirection.Substract(vEye);

	// Calc cross product of Up x Direction
	Vector3f vUp;
	vUp.Set(fUpX, fUpY, fUpZ);
	Vector3f vUpDirection;
	vUpDirection.Set(vDirection);
	vUpDirection.CrossProduct(vUp);

	// Rotate Direction around (Up x Direction)
	Quaternion q;
	Matrix44f m;
	vUpDirection.Normalize();
	q.Set(vUpDirection, fAngle);
	q.Normalize();
	q.ConvertToMatrix(m);
	vDirection.Transform(m);

	// we have new Direction now, calc new Target = Eye + Direction
	Vector3f vTarget;
	vTarget.Set(fEyeX, fEyeY, fEyeZ);
	vTarget.Add(vDirection);

	fTargetX = vTarget.x;
	fTargetY = vTarget.y;
	fTargetZ = vTarget.z;
}

void Camera::ApplyYaw(float fAngle) {
	//
	// Rotate Direction around Up
	//

	// Direction = Target - eye
	Vector3f vDirection;
	vDirection.Set(fTargetX, fTargetY, fTargetZ);
	Vector3f vEye;
	vEye.Set(fEyeX, fEyeY, fEyeZ);
	vDirection.Substract(vEye);

	Vector3f vUp;
	vUp.Set(fUpX, fUpY, fUpZ);

	// Rotate Direction around Up
	Quaternion q;
	Matrix44f m;
	q.Set(vUp, fAngle);
	q.Normalize();
	q.ConvertToMatrix(m);
	vDirection.Transform(m);

	// we have new Direction now, calc new Target = Eye + Direction
	Vector3f vTarget;
	vTarget.Set(fEyeX, fEyeY, fEyeZ);
	vTarget.Add(vDirection);

	fTargetX = vTarget.x;
	fTargetY = vTarget.y;
	fTargetZ = vTarget.z;
}

void Camera::ApplyRoll(float fAngle) {
	//
	// Rotate Up around Direction
	//

	// Direction = Target - eye
	Vector3f vDirection;
	vDirection.Set(fTargetX, fTargetY, fTargetZ);
	Vector3f vEye;
	vEye.Set(fEyeX, fEyeY, fEyeZ);
	vDirection.Substract(vEye);

	Vector3f vUp;
	vUp.Set(fUpX, fUpY, fUpZ);

	// Rotate Up around Direction
	Quaternion q;
	Matrix44f m;
	vDirection.Normalize();
	q.Set(vDirection, fAngle);
	q.Normalize();
	q.ConvertToMatrix(m);
	vUp.Transform(m);
	vUp.Normalize();

	// we have new Up
	fUpX = vUp.x;
	fUpY = vUp.y;
	fUpZ = vUp.z;
}

void Camera::ApplyPan(float fSide, float fUp) {

	// Calc Direction = Target - Eye
	Vector3f vDirection, vEye;
	vDirection.Set(fTargetX, fTargetY, fTargetZ);
	vEye.Set(fEyeX, fEyeY, fEyeZ);
	vDirection.Substract(vEye);

	// Calc cross product of Up x Direction
	Vector3f vUp;
	vUp.Set(fUpX, fUpY, fUpZ);
	Vector3f vUpDirection;
	vUpDirection.Set(vDirection);
	vUpDirection.CrossProduct(vUp);

	// Horizontal: Move eye and Target in direction of Up x Direction
	// Vertical: Move eye and Target in direction of Up

	vUp.Normalize();
	vUpDirection.Normalize();

	vUp.x *= fUp;
	vUp.y *= fUp;
	vUp.z *= fUp;

	vUpDirection.x *= fSide;
	vUpDirection.y *= fSide;
	vUpDirection.z *= fSide;

	// Now merge horizontal and vertical movement into single vector
	vUp.Add(vUpDirection);

	fEyeX += vUp.x;
	fEyeY += vUp.y;
	fEyeZ += vUp.z;

	fTargetX += vUp.x;
	fTargetY += vUp.y;
	fTargetZ += vUp.z;
}

void Camera::ApplyDepth(float fDistance) {

	// Calc Direction = Target - Eye
	Vector3f vDirection, vEye;
	vDirection.Set(fTargetX, fTargetY, fTargetZ);
	vEye.Set(fEyeX, fEyeY, fEyeZ);
	vDirection.Substract(vEye);

	// Move Eye and Target in Direction for fDistance...
	vDirection.Normalize();
	vDirection.x *= fDistance;
	vDirection.y *= fDistance;
	vDirection.z *= fDistance;

	fEyeX += vDirection.x;
	fEyeY += vDirection.y;
	fEyeZ += vDirection.z;
}

void Camera::ApplyPitchOnEye(float fAngle) {

	fAngle = -fAngle;

	//
	// Rotate eye point and Up around Up x Direction
	//

	// Calc Direction = Target - Eye
	Vector3f vDirection, vEye;
	vDirection.Set(fTargetX, fTargetY, fTargetZ);
	vEye.Set(fEyeX, fEyeY, fEyeZ);
	vDirection.Substract(vEye);

	// Calc eye relative to target
	Vector3f vTarget;
	vTarget.Set(fTargetX, fTargetY, fTargetZ);
	vEye.Substract(vTarget);

	// Calc cross product of Up x Direction
	Vector3f vUp;
	vUp.Set(fUpX, fUpY, fUpZ);
	Vector3f vUpDirection;
	vUpDirection.Set(vDirection);
	vUpDirection.CrossProduct(vUp);

	// Rotate RelativeEye around UpDirection
	// and also rotate Up around UpDirection
	Quaternion q;
	Matrix44f m;
	vUpDirection.Normalize();
	q.Set(vUpDirection, fAngle);
	q.Normalize();
	q.ConvertToMatrix(m);
	vEye.Transform(m);
	vUp.Transform(m);

	// Translate eye back according to target
	vEye.Add(vTarget);

	fEyeX = vEye.x;
	fEyeY = vEye.y;
	fEyeZ = vEye.z;

	fUpX = vUp.x;
	fUpY = vUp.y;
	fUpZ = vUp.z;
}

void Camera::ApplyYawOnEye(float fAngle) {

	fAngle = -fAngle;

	//
	// Rotate eye point around Up
	//

	// Calc eye relative to target
	Vector3f vTarget;
	vTarget.Set(fTargetX, fTargetY, fTargetZ);
	Vector3f vEye;
	vEye.Set(fEyeX, fEyeY, fEyeZ);
	vEye.Substract(vTarget);

	// Calc up
	Vector3f vUp;
	vUp.Set(fUpX, fUpY, fUpZ);

	// Rotate RelativeEye around Up
	Quaternion q;
	vUp.Normalize();
	q.Set(vUp, fAngle);
	q.Normalize();
	Matrix44f m;
	q.ConvertToMatrix(m);
	vEye.Transform(m);

	// Translate eye back according to target
	vEye.Add(vTarget);

	fEyeX = vEye.x;
	fEyeY = vEye.y;
	fEyeZ = vEye.z;
}
