/**
 * Camera.cpp
 * Interface for the Camera class.
 *
 * DESCRIPTION:
 *
 * Camera class encapsulates logic to handle camera to view your OpenGL
 * world.
 * OpenGL is using GL_PROJECTION and GL_MODELVIEW matrices to modify
 * the way you view your world. When both matrices are initialized to
 * identity matrices, OpenGL viewport is using some sort of 'camera' placed
 * at center of coordinate system (0, 0, 0) looking down the negative
 * Z-axis, having Y-axis going from bottom to top of viewport,
 * and X-axis from left to right - having center of coordinate system in
 * the middle of the viewport.
 * If you want to use perspective projection with concept of camera
 * (based on so-called modelview duality), you should set parameters of
 * this Camera (you can use Apply* functions to modify them), and
 * call camera.ApplyToGLViewport(). This class encapsulates all logic
 * to handle GL_PROJECTION and GL_MODELVIEW matrices.
 *
 * USAGE:
 *      // Create instance of this camera
 *      // (e.g. member variable of your GLViewport implementation)
 *          Camera camera;
 *
 *      // Set its parameters
 *          camera.SetOrientation(
 *              0.0f, 0.0f, -20.0f, // eye coordinates
 *              0, 0, 0,            // target is center of coordinate system
 *              1, 0, 0             // up is in y-axis direction
 *              );
 *          camera.SetFieldOfView(60.0f);
 *
 *      // Modify it
 *          camera.ApplyYaw(...);
 *
 *      // Apply it on GLViewport::DrawImpl(...) before drawing world
 *          camera.ApplyToGLViewport();
 *          world3d.Draw(...);      // for example..
 *
 * @author Matija Tomaskovic
 * @version 24-Jul-2001
 */

#if !defined(_Camera_H__INCLUDED_)
#define _Camera_H__INCLUDED_

#include <windows.h>
#include <gl\gl.h>              // Header File For The OpenGL32 Library
#include <gl\glu.h>             // Header File For The GLu32 Library

#include "GLTextureManager.h"


class Camera
{
public:
    Camera();
    virtual ~Camera();

    void SetViewportSize(int iWindowWidth, int iWindowHeight);

    void ApplyToGLViewport();

    void SetOrientation(float fEyeX, float fEyeY, float fEyeZ,
                float fTargetX, float fTargetY, float fTargetZ,
                float fUpX, float fUpY, float fUpZ);

    void SetFieldOfView(float fFOV);

    void Draw(GLTextureManager &glTextureManager);

	int iWindowWidth;
	int iWindowHeight;

    float fFOV;

    float fEyeX;
    float fEyeY;
    float fEyeZ;
    float fTargetX;
    float fTargetY;
    float fTargetZ;
    float fUpX;
    float fUpY;
    float fUpZ;

    float fNear;
    float fFar;

	void ApplyDepth(float fDistance);
	void ApplyPan(float fSide, float fUp);
	void ApplyRoll(float fAngle);
	void ApplyYaw(float fAngle);
	void ApplyPitch(float fAngle);
	void ApplyYawOnEye(float fAngle);
	void ApplyPitchOnEye(float fAngle);


};

#endif // !defined(_Camera_H__INCLUDED_)
