/**
 * GLTextureManager.cpp
 *
 * @author Matija Tomaskovic
 * @version 06-Jun-2001
 */

#include <windows.h>            // Header File For Windows
#include <stdio.h>              // Header File For Standard Input/Output ( NEW )

// Must link with: OpenGL32.lib GLu32.lib
#include <gl\gl.h>				// Header File For The OpenGL32 Library
#include <gl\glu.h>				// Header File For The GLu32 Library
#include <gl\glaux.h>           // Header File For The GLaux Library

#include "GLTextureManager.h"
#include <tomaskovic\util\MemoryManager.h>
#include <tomaskovic\util\Vector.h>

typedef struct {
	int iUserID;
	char szFilename[512];
} USER_TEXTURE_ID;

Vector g_vUserTextureIDs;

int g_iGLTextureManagerInstances = 0;


GLTextureManager::GLTextureManager() {
    aTextures.SetElementSize(sizeof(TEXTURE));
    default_glTextureMinFilter = GL_LINEAR;
    default_glTextureMagFilter = GL_LINEAR;
    g_iGLTextureManagerInstances++;
}

GLTextureManager::~GLTextureManager() {

    g_iGLTextureManagerInstances--;

	// Delete GL textures
	for (int i=0; i<aTextures.Size(); i++) {
		TEXTURE* pTexture = (TEXTURE*) aTextures.ElementAt(i);
		glDeleteTextures(1, &pTexture->iGLTextureID);
	}

	// If no more texture managers - clear user texture IDs
    if (g_iGLTextureManagerInstances == 0) {
		for (int i=0; i<g_vUserTextureIDs.Size(); i++) {
			USER_TEXTURE_ID* p = (USER_TEXTURE_ID*) g_vUserTextureIDs.ElementAt(i);
			delete p;
		}
	}
}


TEXTURE* GLTextureManager::LoadTextureFromBMP(char* pszFilename) {

	int iTextureID = -1;

    FILE *pFile = fopen(pszFilename, "r");

    if (!pFile)
        return NULL;

	fclose(pFile);

    AUX_RGBImageRec *pTextureImage = auxDIBImageLoad(pszFilename);

    TEXTURE *pTexture = (TEXTURE*) aTextures.AddElement();
    strcpy(pTexture->szFilename, pszFilename);
    pTexture->iWidth = pTextureImage->sizeX;
    pTexture->iHeight = pTextureImage->sizeY;
    glGenTextures(1, &pTexture->iGLTextureID);

	// Typical Texture Generation Using Data From The Bitmap
    glBindTexture(GL_TEXTURE_2D, pTexture->iGLTextureID);
    glTexImage2D(GL_TEXTURE_2D, 0, 3,
        pTexture->iWidth, pTexture->iHeight,
        0, GL_RGB, GL_UNSIGNED_BYTE, pTextureImage->data);

    ApplyTextureDefaultOptions();

    if (pTextureImage->data) {
        MemFree(pTextureImage->data);
    }
	MemFree(pTextureImage);

    return pTexture;
}

GLuint GLTextureManager::LoadTextureFromBMPWithMask(char* pszFilename, char* pszMaskFilename) {
/*
    BITMAPINFO *pInfo;
    GLubyte *pBits = LoadDIBitmap(pszFilename, &pInfo);
    if (pBits == NULL)
        return 0;

    int i;
    #ifdef WIN32
        // Convert bitmap from BGR to RGB
        GLubyte *ptr;
        for (i = pInfo->bmiHeader.biWidth * pInfo->bmiHeader.biHeight, ptr = pBits;
            i > 0; i--, ptr += 3)
        {
            // Swap red and blue
            GLubyte temp = ptr[0];
            ptr[0] = ptr[2];
            ptr[2] = temp;
        }
    #endif // WIN32

    BITMAPINFO *pMaskInfo;
    GLubyte *pMaskBits = LoadDIBitmap(pszMaskFilename, &pMaskInfo);
    if (pMaskBits == NULL) {
        MemFree(pInfo);
        MemFree(pBits);
        return 0;
    }

    GLubyte* pRGBABits = MemAlloc(pInfo->bmiHeader.biWidth * pInfo->bmiHeader.biHeight * 4);
    GLubyte* pRGBAPtr, pMaskBitsPtr;
    for (i = pInfo->bmiHeader.biWidth * pInfo->bmiHeader.biHeight,
        ptr = pBits, pRGBAPtr = pRGBABits, pMaskBitsPtr = pMaskBits;
        i > 0; pRGBAPtr += 4, ptr += 3, pMaskBitsPtr += 3)
    {
        pRGBAPtr[0] = ptr[0];
        pRGBAPtr[1] = ptr[1];
        pRGBAPtr[2] = ptr[2];
        pRGBAPtr[3] = pMaskBitsPtr[0];
    }


    TEXTURE *pTexture = (TEXTURE*) aTextures.AddElement();
    strcpy(pTexture->szFilename, pszFilename)
    pTexture->iWidth = pInfo->bmiHeader.biWidth;
    pTexture->iHeight = pInfo->bmiHeader.biHeight;
    glGenTextures(1, &pTexture->iTextureID);

    glBindTexture(GL_TEXTURE_2D, pTexture->iTextureID);

    glTextImage2D(GL_TEXTURE_2D, 0, 3,
        pInfo->bmiHeader.biWidth, pInfo->bmiHeader.biHeight,
        0, GL_RGBA, GL_UNSIGNED_BYTE, pBits);

    MemFree(pInfo);
    MemFree(pBits);
    MemFree(pMaskInfo);
    MemFree(pMaskBits);
    MemFree(pRGBABits);

    ApplyTextureDefaultOptions()

    return pTexture->iTextureID;
 */
    return -1;
}


void GLTextureManager::ApplyTextureDefaultOptions() {
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, default_glTextureMinFilter);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, default_glTextureMagFilter);
}



/**
 * @return user texture ID for texture in given filename (shared between all texture managers)
 */
int GLTextureManager::FindUserTextureID(char *pszTexturePathFilename) {
	int iMaxUserTextureID = 1;
	for (int i=0; i<g_vUserTextureIDs.Size(); i++) {
		USER_TEXTURE_ID* pUserTextureID = (USER_TEXTURE_ID*) g_vUserTextureIDs.ElementAt(i);
		if (strcmp(pUserTextureID->szFilename, pszTexturePathFilename) == 0)
			return pUserTextureID->iUserID;
		if ((i == 0) || (pUserTextureID->iUserID > iMaxUserTextureID))
			iMaxUserTextureID = pUserTextureID->iUserID+1;
	}
	USER_TEXTURE_ID* pUserTextureID = (USER_TEXTURE_ID*) MemAlloc(sizeof(USER_TEXTURE_ID));
	pUserTextureID->iUserID = iMaxUserTextureID;
	strcpy(pUserTextureID->szFilename, pszTexturePathFilename);
	g_vUserTextureIDs.AddElement(pUserTextureID);
	return pUserTextureID->iUserID;
}

int GLTextureManager::FindGLTextureID(int iUserTextureID) {

	// Find global user ID (should be here)
	USER_TEXTURE_ID* pUserTextureID = NULL;
	for (int i=0; i<g_vUserTextureIDs.Size(); i++) {
		USER_TEXTURE_ID* p = (USER_TEXTURE_ID*) g_vUserTextureIDs.ElementAt(i);
		if (p->iUserID == iUserTextureID) {
			pUserTextureID = p;
			break;
		}
	}

	// No global ID?
	if (!pUserTextureID)
		return -1;

	// Loaded already?
	for (i=0; i<aTextures.Size(); i++) {
		TEXTURE* pTexture = (TEXTURE*) aTextures.ElementAt(i);
		if (pTexture->iUserTextureID == iUserTextureID)
			return pTexture->iGLTextureID;
	}

    TEXTURE *pTexture = LoadTextureFromBMP(pUserTextureID->szFilename);
	pTexture->iUserTextureID = iUserTextureID;
	return pTexture->iGLTextureID;
}
