/**
 * GLTextureManager.h
 *
 * GLTextureManager is texture manager used in context of GLRenderingContext.
 * Since textures can not be shared between different OpenGL rendering contexts,
 * each GLRenderingContext has its own GLTextureManager. You will get
 * reference to GLTextureManager on DrawImpl(...) of your GLViewport.
 * You should use FindUserTextureID(pszTexturePathFilename) to get user ID of
 * your bitmap file to be loaded as texture. (user IDs are shared between
 * all GLTextureManagers).
 * Later, you should use FindGLTextureID(iUserTextureID) to get OpenGL
 * texture ID (GLTextureManager will do loading here if texture file is
 * not yet loaded, also, it will apply min and mag filters).
 *
 * @author Matija Tomaskovic
 * @version 07-Aug-2001
 */

#if !defined(_GLTextureManager_H__INCLUDED_)
#define _GLTextureManager_H__INCLUDED_

#include <windows.h>
#include <gl\gl.h>              // Header File For The OpenGL32 Library
#include <gl\glu.h>             // Header File For The GLu32 Library

#include <tomaskovic\util\Array.h>

typedef struct {
	int iUserTextureID;		// User texture ID
    GLuint iGLTextureID;	// OpenGL texture ID, -1 if no one yet
    int iWidth;
    int iHeight;
    char szFilename[512];
} TEXTURE;

class GLTextureManager
{
public:
	int FindGLTextureID(int iUserTextureID);
	static int FindUserTextureID(char* pszTexturePathFilename);

    GLTextureManager();
    ~GLTextureManager();

    /**
     * Loads texture from specified bitmap filename
     * Notes:
     * - bitmap width and height should be power of 2
     */
    TEXTURE* LoadTextureFromBMP(char* pszFilename);

    /**
     * Loads texture from specified bitmap filename
     * Notes:
     * - bitmap width and height should be power of 2
     * - mask bitmap should be RGB bitmap but with grayscale colors
     */
    GLuint LoadTextureFromBMPWithMask(char* pszFilename, char* pszMaskFilename);

	/**
	 * Applies texture parameters to OpenGL state machine
	 * (texture filters,...)
	 */
    void ApplyTextureDefaultOptions();


private:
    Array aTextures;

    GLint default_glTextureMinFilter;   // GL_LINEAR / GL_NEAREST
    GLint default_glTextureMagFilter;
};

#endif // !defined(_GLTextureManager_H__INCLUDED_)
