/**
 * WavefrontObject3dModel.cpp
 *
 * @author Matija Tomaskovic
 * @version 04-Jun-2001
 */

#include "WavefrontObject3dModel.h"
#include <tomaskovic\util\FileInputStream.h>
#include <tomaskovic\util\TextInputReader.h>
#include <tomaskovic\util\SuperFilename.h>

WavefrontObject3dModel::WavefrontObject3dModel() {
    aGeometricVertices.SetElementSize(sizeof(WF_GEOMETRIC_VERTEX));
    aTextureVertices.SetElementSize(sizeof(WF_TEXTURE_VERTEX));
    aVertexNormals.SetElementSize(sizeof(WF_VERTEX_NORMAL));
    aTriangleFaces.SetElementSize(sizeof(WF_TRIANGLE_FACE));
    aMaterials.SetElementSize(sizeof(WF_MATERIAL));
	aGroups.SetElementSize(sizeof(WF_GROUP));

	// Add zero geometric, normal and texture vertices - they are not used
	aGeometricVertices.AddElement();
	aTextureVertices.AddElement();
	aVertexNormals.AddElement();
}

WavefrontObject3dModel::~WavefrontObject3dModel() {
}

BOOL WavefrontObject3dModel::LoadFromFile(char* pszObjFilename) {

	/**
	# Wavefront OBJ ...

	mtllib player1.mtl

	v -3.750000 16.919088 5.250000
	v -10.000000 23.919088 0.000000
	v -4.000000 30.919088 0.000000
	v 3.000000 22.919088 0.000000
	v 3.250000 11.919088 0.000000
	v -11.000000 11.669088 0.000000
	# 6 vertices

	vt 0.734375 0.761719
	# 1 texture coordinates

	vn 0.297172 -0.319848 0.899659
	vn 0.012556 -0.715667 0.698329
	vn 0.235086 -0.527846 0.816158
	vn 0.378232 0.330953 0.864529
	vn 0.000000 0.000000 0.000000
	vn 0.502207 0.175860 0.846677
	# 6 normals

	g Triangles
	usemtl Material03
	s 1
	f 1/1/1 6/1/2 5/1/3
	f 3/1/4 1/1/1 2/1/5
	f 6/1/2 5/1/3 1/1/1
	f 6/1/2 2/1/5 1/1/1
	f 5/1/3 4/1/6 1/1/1
	f 4/1/6 3/1/4 1/1/1
	f 3/1/4 2/1/5 1/1/1
	f 2/1/5 6/1/2 1/1/1
	# 8 triangles in group

	# 8 triangles total
	*/
	FileInputStream fis;

	SuperFilename sf;
	sf.SetPathFilename(pszObjFilename);
	strName.Set(&sf.strFilename);

    if (!fis.Open(pszObjFilename))
        return FALSE;

    TextInputReader tir(&fis);

    while(TRUE) {

        if (!tir.ReadNextWord())
            break;

        if (tir.strLastWord.GetLength() == 0)
            continue;

        if (tir.strLastWord.CharAt(0) == '#') {
            tir.SkipUntilEOL();
            continue;
        }

        if (tir.strLastWord.Equals("v")) {
            WF_GEOMETRIC_VERTEX *p =
                (WF_GEOMETRIC_VERTEX*) aGeometricVertices.AddElement();
            p->x = UNDEFINED_FLOAT_VALUE;
            p->y = UNDEFINED_FLOAT_VALUE;
            p->z = UNDEFINED_FLOAT_VALUE;
            p->w = UNDEFINED_FLOAT_VALUE;
            if (tir.ReadNextWordInLine()) {
                p->x = tir.strLastWord.FloatValue();
                if (tir.ReadNextWordInLine()) {
                    p->y = tir.strLastWord.FloatValue();
                    if (tir.ReadNextWordInLine()) {
                        p->z = tir.strLastWord.FloatValue();
                        if (tir.ReadNextWordInLine()) {
                            p->w = tir.strLastWord.FloatValue();
                        }
                    }
                }
            }
            tir.SkipSpacesAndEOL();
        }
        else if (tir.strLastWord.Equals("vt")) {
            WF_TEXTURE_VERTEX *p =
                (WF_TEXTURE_VERTEX*) aTextureVertices.AddElement();
            p->u = UNDEFINED_FLOAT_VALUE;
            p->v = UNDEFINED_FLOAT_VALUE;
            p->w = UNDEFINED_FLOAT_VALUE;
            if (tir.ReadNextWordInLine()) {
                p->u = tir.strLastWord.FloatValue();
                if (tir.ReadNextWordInLine()) {
                    p->v = tir.strLastWord.FloatValue();
                    if (tir.ReadNextWordInLine()) {
                        p->w = tir.strLastWord.FloatValue();
                    }
                }
            }
            tir.SkipSpacesAndEOL();
        }
        else if (tir.strLastWord.Equals("vn")) {
            WF_VERTEX_NORMAL *p =
                (WF_VERTEX_NORMAL*) aVertexNormals.AddElement();
            p->i = UNDEFINED_FLOAT_VALUE;
            p->j = UNDEFINED_FLOAT_VALUE;
            p->k = UNDEFINED_FLOAT_VALUE;
            if (tir.ReadNextWordInLine()) {
                p->i = tir.strLastWord.FloatValue();
                if (tir.ReadNextWordInLine()) {
                    p->j = tir.strLastWord.FloatValue();
                    if (tir.ReadNextWordInLine()) {
                        p->k = tir.strLastWord.FloatValue();
                    }
                }
            }
            tir.SkipSpacesAndEOL();
        }
        else if (tir.strLastWord.Equals("mtllib")) {

            /***
             * mtllib player1.mtl
             **/

            if (!tir.ReadNextWord()) return FALSE;

            SuperFilename filenameMaterial;
            filenameMaterial.SetPathFilename(pszObjFilename);
            filenameMaterial.SetFilename(tir.strLastWord.GetBuffer());
            if (!LoadMaterialsFromLibrary(
                filenameMaterial.strPathFilename.GetBuffer()))
            {
                return FALSE;
            }

        }
        else if (tir.strLastWord.Equals("g")) {
            /*
                g Triangles
                usemtl Material03
                s 1
                f 1/1/1 6/1/2 5/1/3
                f 3/1/4 1/1/1 2/1/5
                f 6/1/2 5/1/3 1/1/1
                f 6/1/2 2/1/5 1/1/1
                f 5/1/3 4/1/6 1/1/1
                f 4/1/6 3/1/4 1/1/1
                f 3/1/4 2/1/5 1/1/1
                f 2/1/5 6/1/2 1/1/1
                # 8 triangles in group
             */
            WF_GROUP *pGroup =
                (WF_GROUP*) aGroups.AddElement();
            pGroup->szName[0] = 0;
            pGroup->iFirstFaceIndex = -1;
			pGroup->pMaterial = NULL;
			pGroup->szMaterialName[0] = 0;
			pGroup->iFaces = 0;

            if (!tir.ReadNextWordInLine()) return FALSE;
            strcpy(pGroup->szName, tir.strLastWord.GetBuffer());

            tir.SkipSpacesAndEOL();

            // usemtl Material03
            if (!tir.ReadNextWordInLine("usemtl")) return FALSE;
            if (!tir.ReadNextWordInLine()) return FALSE;
            strcpy(pGroup->szMaterialName, tir.strLastWord.GetBuffer());
            tir.SkipSpacesAndEOL();

            // Skip 's 1' definition... (smoothing group)
            if (!tir.ReadNextWordInLine()) return FALSE;
            if (tir.strLastWord.Equals("s")) {
                if (!tir.ReadNextWordInLine()) return FALSE;
                tir.SkipSpacesAndEOL();
            }

            //
            // Read faces
            //

            while(TRUE) {

                if (!tir.ReadNextWordInLine("f")) {
                    tir.PushBack((BYTE*) tir.strLastWord.GetBuffer(),
                        tir.strLastWord.GetLength());
                    break;
                }

                WF_TRIANGLE_FACE *pFace =
                    (WF_TRIANGLE_FACE*) aTriangleFaces.AddElement();

                if (pGroup->iFirstFaceIndex == -1)
                    pGroup->iFirstFaceIndex = aTriangleFaces.Size() - 1;

                tir.SkipSpaces();
                int v1 = UNDEFINED_FLOAT_VALUE;
                int vt1 = UNDEFINED_FLOAT_VALUE;
                int vn1 = UNDEFINED_FLOAT_VALUE;
                int v2 = UNDEFINED_FLOAT_VALUE;
                int vt2 = UNDEFINED_FLOAT_VALUE;
                int vn2 = UNDEFINED_FLOAT_VALUE;
                int v3 = UNDEFINED_FLOAT_VALUE;
                int vt3 = UNDEFINED_FLOAT_VALUE;
                int vn3 = UNDEFINED_FLOAT_VALUE;

                // Read first vertex
                if (tir.ReadWordUntilDelimiterAndSkipDelimiter('/')) {
                    v1 = tir.strLastWord.IntValue();
                    if (tir.ReadWordUntilDelimiterAndSkipDelimiter('/')) {
                        vt1 = tir.strLastWord.IntValue();
                        if (tir.ReadWordUntilDelimiterAndSkipDelimiter('/')) {
                            vn1 = tir.strLastWord.IntValue();
                        }
                    }
                }
                tir.SkipSpaces();

                // Read second vertex
                if (tir.ReadWordUntilDelimiterAndSkipDelimiter('/')) {
                    v2 = tir.strLastWord.IntValue();
                    if (tir.ReadWordUntilDelimiterAndSkipDelimiter('/')) {
                        vt2 = tir.strLastWord.IntValue();
                        if (tir.ReadWordUntilDelimiterAndSkipDelimiter('/')) {
                            vn2 = tir.strLastWord.IntValue();
                        }
                    }
                }
                tir.SkipSpaces();

                // Read third vertex
                if (tir.ReadWordUntilDelimiterAndSkipDelimiter('/')) {
                    v3 = tir.strLastWord.IntValue();
                    if (tir.ReadWordUntilDelimiterAndSkipDelimiter('/')) {
                        vt3 = tir.strLastWord.IntValue();
                        if (tir.ReadWordUntilDelimiterAndSkipDelimiter('/')) {
                            vn3 = tir.strLastWord.IntValue();
                        }
                    }
                }
                tir.SkipSpacesAndEOL();

                pFace->iGeometricVertexIndex[0] = v1;
                pFace->iTextureVertexIndex[0] = vt1;
                pFace->iVertexNormalIndex[0] = vn1;

                pFace->iGeometricVertexIndex[1] = v2;
                pFace->iTextureVertexIndex[1] = vt2;
                pFace->iVertexNormalIndex[1] = vn2;

                pFace->iGeometricVertexIndex[2] = v3;
                pFace->iTextureVertexIndex[2] = vt3;
                pFace->iVertexNormalIndex[2] = vn3;

				pGroup->iFaces++;
            }
        }
        else {
            return FALSE;
        }
    }

    fis.Close();

	// Connect groups with their materials
	for (int i=0; i<aGroups.Size(); i++) {
		WF_GROUP* pGroup = (WF_GROUP*) aGroups.ElementAt(i);
		for (int t=0; t<aMaterials.Size(); t++) {
			WF_MATERIAL* pMaterial =
				(WF_MATERIAL*) aMaterials.ElementAt(t);
			if (strcmp(pGroup->szMaterialName, pMaterial->szName) == 0) {
				pGroup->pMaterial = pMaterial;
				break;
			}
		}
	}

	return TRUE;
}



BOOL WavefrontObject3dModel::LoadMaterialsFromLibrary(char* pszMtlFilename) {

	/***
	#
	# player1.mtl
	#

	newmtl Material01
	Ka 0.200000 0.200000 0.200000
	Kd 0.800000 0.800000 0.800000
	Ks 0.000000 0.000000 0.000000
	Ns 0.000000
	map_Kd .\fifa_stadium 213.jpg

	newmtl Material02
	Ka 0.317647 0.337255 0.062745
	Kd 0.596078 0.184314 0.815686
	Ks 0.937255 0.627451 0.960784
	Ns 132.812500
	*/

	FileInputStream fisMaterial;

    if (!fisMaterial.Open(pszMtlFilename))
        return FALSE;

    TextInputReader tirMaterial(&fisMaterial);

    while(TRUE) {

        if (!tirMaterial.ReadNextWord())
            break;

        // Empty row?
        if (tirMaterial.strLastWord.GetLength() == 0)
            continue;

        // Ignore comments
        if (tirMaterial.strLastWord.CharAt(0) == '#') {
            tirMaterial.SkipUntilEOL();
            continue;
        }

        if (tirMaterial.strLastWord.Equals("newmtl")) {

            // Read material name
            if (!tirMaterial.ReadNextWordInLine()) return FALSE;

            WF_MATERIAL *p =
                (WF_MATERIAL*) aMaterials.AddElement();
            strcpy(p->szName, tirMaterial.strLastWord.GetBuffer());
            p->iUserTextureID = -1;

            // Get into new row
            tirMaterial.SkipSpacesAndEOL();

            if (!tirMaterial.ReadNextWordInLine("Ka")) return FALSE;

            if (!tirMaterial.ReadNextWordInLine()) return FALSE;
            p->colorAmbient.fRed = tirMaterial.strLastWord.FloatValue();

            if (!tirMaterial.ReadNextWordInLine()) return FALSE;
            p->colorAmbient.fGreen = tirMaterial.strLastWord.FloatValue();

            if (!tirMaterial.ReadNextWordInLine()) return FALSE;
            p->colorAmbient.fBlue = tirMaterial.strLastWord.FloatValue();


            tirMaterial.SkipSpacesAndEOL();
            if (!tirMaterial.ReadNextWordInLine("Kd")) return FALSE;

            if (!tirMaterial.ReadNextWordInLine()) return FALSE;
            p->colorDiffuse.fRed = tirMaterial.strLastWord.FloatValue();

            if (!tirMaterial.ReadNextWordInLine()) return FALSE;
            p->colorDiffuse.fGreen = tirMaterial.strLastWord.FloatValue();

            if (!tirMaterial.ReadNextWordInLine()) return FALSE;
            p->colorDiffuse.fBlue = tirMaterial.strLastWord.FloatValue();


            tirMaterial.SkipSpacesAndEOL();
            if (!tirMaterial.ReadNextWordInLine("Ks")) return FALSE;

            if (!tirMaterial.ReadNextWordInLine()) return FALSE;
            p->colorSpecular.fRed = tirMaterial.strLastWord.FloatValue();

            if (!tirMaterial.ReadNextWordInLine()) return FALSE;
            p->colorSpecular.fGreen = tirMaterial.strLastWord.FloatValue();

            if (!tirMaterial.ReadNextWordInLine()) return FALSE;
            p->colorSpecular.fBlue = tirMaterial.strLastWord.FloatValue();


            tirMaterial.SkipSpacesAndEOL();
            if (!tirMaterial.ReadNextWordInLine("Ns")) return FALSE;

            if (!tirMaterial.ReadNextWordInLine()) return FALSE;
            p->fShinnes = tirMaterial.strLastWord.FloatValue();


            tirMaterial.SkipSpacesAndEOL();
            if (!tirMaterial.ReadNextWordInLine()) return TRUE;

            if (tirMaterial.strLastWord.Equals("map_Kd")) {
                p->szDiffuseMap[0] = 0;
                if (!tirMaterial.ReadNextWord()) return TRUE;

                SuperFilename filenameMap;
                filenameMap.SetPathFilename(pszMtlFilename);
                filenameMap.MergeRelativeFilename(tirMaterial.strLastWord.GetBuffer());
                strcpy(p->szDiffuseMap, filenameMap.strPathFilename.GetBuffer());
            }
            else {
                tirMaterial.PushBack((BYTE*) tirMaterial.strLastWord.GetBuffer(),
                    tirMaterial.strLastWord.GetLength());
                continue;
            }
        }
        else {
            return FALSE;
        }
    }

	return TRUE;
}

void WavefrontObject3dModel::Draw(GLTextureManager &textureManager) {

	glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);

    // For each group
    for (int i=0; i<aGroups.Size(); i++) {
        WF_GROUP* pGroup = (WF_GROUP*) aGroups.ElementAt(i);
		if (pGroup->iFirstFaceIndex == -1)
			return;

        WF_MATERIAL* pMaterial = pGroup->pMaterial;
        WF_GEOMETRIC_VERTEX* pVertex1;
		WF_GEOMETRIC_VERTEX* pVertex2;
		WF_GEOMETRIC_VERTEX* pVertex3;


		glColor4f(1.0f, 1.0f, 1.0f, 1.0f);

		//
		// Find GL texture ID
		//

		int iGLTextureID = -1;
		if (pMaterial) {
			// Assign user texture ID to this material, if not yet assigned..
			if (pMaterial->iUserTextureID == -1)
                pMaterial->iUserTextureID = GLTextureManager::FindUserTextureID(pMaterial->szDiffuseMap);
			iGLTextureID = textureManager.FindGLTextureID(pMaterial->iUserTextureID);
		}

		if (iGLTextureID != -1)
			glEnable(GL_TEXTURE_2D);
		else
			glDisable(GL_TEXTURE_2D);

        // Paint every triangle face
        glBegin(GL_TRIANGLES);

        if ((pMaterial) && (iGLTextureID != -1)) {

            glBindTexture(GL_TEXTURE_2D, iGLTextureID);
			glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
			glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

            for (int t=pGroup->iFirstFaceIndex;
                t<pGroup->iFirstFaceIndex + pGroup->iFaces;
                t++)
            {
                // Providing opengl with 3 vertices
                WF_TRIANGLE_FACE* pFace =
                    (WF_TRIANGLE_FACE*) aTriangleFaces.ElementAt(t);

                pVertex1 = (WF_GEOMETRIC_VERTEX*) aGeometricVertices.ElementAt(pFace->iGeometricVertexIndex[0]);
                pVertex2 = (WF_GEOMETRIC_VERTEX*) aGeometricVertices.ElementAt(pFace->iGeometricVertexIndex[1]);
                pVertex3 = (WF_GEOMETRIC_VERTEX*) aGeometricVertices.ElementAt(pFace->iGeometricVertexIndex[2]);

                int iTextureVertexIndex1 = pFace->iTextureVertexIndex[0];
                int iTextureVertexIndex2 = pFace->iTextureVertexIndex[1];
                int iTextureVertexIndex3 = pFace->iTextureVertexIndex[2];

                if (iTextureVertexIndex1 != UNDEFINED_FLOAT_VALUE) {
                    WF_TEXTURE_VERTEX* pTVertex1 =
						(WF_TEXTURE_VERTEX*) aTextureVertices.ElementAt(pFace->iTextureVertexIndex[0]);
                    glTexCoord2f(pTVertex1->u, pTVertex1->v);
                }
                glVertex3f(pVertex1->x, pVertex1->y, pVertex1->z);

                if (iTextureVertexIndex2 != UNDEFINED_FLOAT_VALUE) {
                    WF_TEXTURE_VERTEX* pTVertex2 =
						(WF_TEXTURE_VERTEX*) aTextureVertices.ElementAt(pFace->iTextureVertexIndex[1]);
                    glTexCoord2f(pTVertex2->u, pTVertex2->v);
                }
                glVertex3f(pVertex2->x, pVertex2->y, pVertex2->z);

                if (iTextureVertexIndex3 != UNDEFINED_FLOAT_VALUE) {
                    WF_TEXTURE_VERTEX* pTVertex3 =
						(WF_TEXTURE_VERTEX*) aTextureVertices.ElementAt(pFace->iTextureVertexIndex[2]);
                    glTexCoord2f(pTVertex3->u, pTVertex3->v);
                }
                glVertex3f(pVertex3->x, pVertex3->y, pVertex3->z);
            }
        }
        else {
            for (int t=pGroup->iFirstFaceIndex;
                t<pGroup->iFirstFaceIndex + pGroup->iFaces;
                t++)
            {
                // Providing opengl with 3 vertices
                WF_TRIANGLE_FACE* pFace =
                    (WF_TRIANGLE_FACE*) aTriangleFaces.ElementAt(t);

				glColor4f(1.0f, 1.0f, 1.0f, 1.0f);

                pVertex1 = (WF_GEOMETRIC_VERTEX*) aGeometricVertices.ElementAt(pFace->iGeometricVertexIndex[0]);
                pVertex2 = (WF_GEOMETRIC_VERTEX*) aGeometricVertices.ElementAt(pFace->iGeometricVertexIndex[1]);
                pVertex3 = (WF_GEOMETRIC_VERTEX*) aGeometricVertices.ElementAt(pFace->iGeometricVertexIndex[2]);
                glVertex3f(pVertex1->x, pVertex1->y, pVertex1->z);
                glVertex3f(pVertex2->x, pVertex2->y, pVertex2->z);
                glVertex3f(pVertex3->x, pVertex3->y, pVertex3->z);
            }
        }

        glEnd();

		glDisable(GL_TEXTURE_2D);

    }
}


void WavefrontObject3dModel::DrawWireframe() {

    // For each group
    for (int i=0; i<aGroups.Size(); i++) {
        WF_GROUP* pGroup = (WF_GROUP*) aGroups.ElementAt(i);
		if (pGroup->iFirstFaceIndex == -1)
			return;

		glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);

        WF_MATERIAL* pMaterial = pGroup->pMaterial;
        WF_GEOMETRIC_VERTEX* pVertex1;
		WF_GEOMETRIC_VERTEX* pVertex2;
		WF_GEOMETRIC_VERTEX* pVertex3;

        // Paint every triangle face
        glBegin(GL_TRIANGLES);

        for (int t=pGroup->iFirstFaceIndex;
            t<pGroup->iFirstFaceIndex + pGroup->iFaces;
            t++)
        {
            // Providing opengl with 3 vertices
            WF_TRIANGLE_FACE* pFace =
                (WF_TRIANGLE_FACE*) aTriangleFaces.ElementAt(t);

			glColor4f(1.0f, 1.0f, 1.0f, 1.0f);

            pVertex1 = (WF_GEOMETRIC_VERTEX*) aGeometricVertices.ElementAt(pFace->iGeometricVertexIndex[0]);
            pVertex2 = (WF_GEOMETRIC_VERTEX*) aGeometricVertices.ElementAt(pFace->iGeometricVertexIndex[1]);
            pVertex3 = (WF_GEOMETRIC_VERTEX*) aGeometricVertices.ElementAt(pFace->iGeometricVertexIndex[2]);
            glVertex3f(pVertex1->x, pVertex1->y, pVertex1->z);
            glVertex3f(pVertex2->x, pVertex2->y, pVertex2->z);
            glVertex3f(pVertex3->x, pVertex3->y, pVertex3->z);
        }

        glEnd();
    }
}


void WavefrontObject3dModel::GetBounds(float *pMinX, float *pMaxX, float *pMinY, float *pMaxY, float *pMinZ, float *pMaxZ)
{
	float fMinX, fMaxX, fMinY, fMaxY, fMinZ, fMaxZ;

	for (int i=1; i<aGeometricVertices.Size(); i++) {
		WF_GEOMETRIC_VERTEX* pVertex =
			(WF_GEOMETRIC_VERTEX*) aGeometricVertices.ElementAt(i);

		if (i == 1) {
			fMinX = pVertex->x;
			fMaxX = pVertex->x;
			fMinY = pVertex->y;
			fMaxY = pVertex->y;
			fMinZ = pVertex->z;
			fMaxZ = pVertex->z;
		}
		else {
			if (pVertex->x < fMinX) fMinX = pVertex->x;
			if (pVertex->x > fMaxX) fMaxX = pVertex->x;
			if (pVertex->y < fMinY) fMinY = pVertex->y;
			if (pVertex->y > fMaxY) fMaxY = pVertex->y;
			if (pVertex->z < fMinZ) fMinZ = pVertex->z;
			if (pVertex->z > fMaxZ) fMaxZ = pVertex->z;
		}
	}

	*pMinX = fMinX;
	*pMaxX = fMaxX;
	*pMinY = fMinY;
	*pMaxY = fMaxY;
	*pMinZ = fMinZ;
	*pMaxZ = fMaxZ;
}
