/**
 * World3dModel.cpp
 * Implementation of the World3dModel class.
 *
 * @author Matija Tomaskovic
 * @version 21-May-2001
 */

#include <tomaskovic\util\Buffer.h>
#include <tomaskovic\util\TextWriter.h>
#include <tomaskovic\util\TextInputReader.h>
#include <tomaskovic\util\SuperString.h>
#include <tomaskovic\util\MemoryManager.h>
#include <tomaskovic\util\SuperFilename.h>
#include <tomaskovic\xml\XMLDocument.h>
#include <tomaskovic\xml\XMLElement.h>
#include <tomaskovic\xml\XMLAttribute.h>

#include "World3dModel.h"
#include "WavefrontObject3dModel.h"
#include "World3dModelCamera.h"
#include "World3dModelCameraKeyframe.h"
#include "World3dModelModelInstance.h"
#include "World3dModelModelInstanceKeyframe.h"
#include "World3dModelFrameObject.h"
#include "World3dModelKeyframe.h"
#include "Autodesk3dsDatabase.h"
#include "Autodesk3dsObject3dModel.h"
#include "ModelObject3d.h"


World3dModel::World3dModel() {
}


World3dModel::~World3dModel() {

	int i;

	// Delete all frame objects
	for (i=0; i<vFrameObjects.Size(); i++) {
		World3dModelFrameObject* pFrameObject = 
			(World3dModelFrameObject*) vFrameObjects.ElementAt(i);
		delete pFrameObject;
	}

	// Delete all frame names
	for (i=0; i<vFrameNames.Size(); i++) {
		W3D_FRAME_NAME* p = 
			(W3D_FRAME_NAME*) vFrameNames.ElementAt(i);
		delete p;
	}

	// Delete all model definitions
	for (i=0; i<vModelDefinitions.Size(); i++) {
		W3D_MODEL_DEFINITION* pModelDefinition = 
			(W3D_MODEL_DEFINITION*) vModelDefinitions.ElementAt(i);
		if (pModelDefinition->pObject3dModel)
			delete pModelDefinition->pObject3dModel;
		delete pModelDefinition;
	}

	// Delete all autodesk 3ds databases
	for (i=0; i<vAutodesk3dsDatabases.Size(); i++) {
		Autodesk3dsDatabase* pAutodesk3dsDatabase = 
			(Autodesk3dsDatabase*) vAutodesk3dsDatabases.ElementAt(i);
		delete pAutodesk3dsDatabase;
	}


}


void World3dModel::Save(OutputStream* pOutputStream) {

	XMLDocument doc;
	int i;

	XMLElement *pEWorld3dModel = 
		new XMLElement("world3dmodel");
	doc.pRootElement = pEWorld3dModel;

	//
	// Save model definitions
	//

	XMLElement* pElementModelDefinitions = 
		pEWorld3dModel->AddChildElement("model-definitions");
	for (i=0; i<vModelDefinitions.Size(); i++) {
		W3D_MODEL_DEFINITION* pModelDefinition =
			(W3D_MODEL_DEFINITION*) vModelDefinitions.ElementAt(i);
		XMLElement* p = SaveModelDefinition(pModelDefinition, strPathFilename.GetBuffer());
		pElementModelDefinitions->AddChildElement(p);
	}

	//
	// Save frame names
	//

	XMLElement* pElementFrameNames = 
		pEWorld3dModel->AddChildElement("frame-names");
	for (i=0; i<vFrameNames.Size(); i++) {
		W3D_FRAME_NAME* pFrameName =
			(W3D_FRAME_NAME*) vFrameNames.ElementAt(i);
		XMLElement * p = pElementFrameNames->AddChildElement("frame-name");
		p->AddAttribute("number", pFrameName->iFrameNo);
		p->AddAttribute("name", pFrameName->szName);
	}

	//
	// Save frame objects
	//

	XMLElement* pECameras = 
		pEWorld3dModel->AddChildElement("cameras");
	XMLElement* pEModelInstances = 
		pEWorld3dModel->AddChildElement("model-instances");
	for (i=0; i<vFrameObjects.Size(); i++) {
		World3dModelFrameObject* pFrameObject =
			(World3dModelFrameObject*) vFrameObjects.ElementAt(i);
		if (pFrameObject->strType.Equals("Camera")) {
			World3dModelCamera* pCamera = 
				(World3dModelCamera*) pFrameObject;
			XMLElement* p = pCamera->Save();
			pECameras->AddChildElement(p);
		}
		else if (pFrameObject->strType.Equals("ModelInstance")) {
			World3dModelModelInstance* pModelInstance = 
				(World3dModelModelInstance*) pFrameObject;
			XMLElement* p = pModelInstance->Save();
			pEModelInstances->AddChildElement(p);
		}
		else {
		}
	}

	doc.Save(pOutputStream);
}


BOOL World3dModel::Load(InputStream* pInputStream) {

	/*
<?xml version="1.0" encoding="ISO-8859-2"?>
<world3dmodel>
    <model-definitions>
        <model-definition name="quad.obj" type="WAVEFRONT" filename=".\quad.obj"/>
    </model-definitions>
    <frame-names/>
    <model-instances>
        <model-instance instance-name="q1" model-name="quad.obj">
            <model-instance-keyframe frame="20" pos-x="0.0" pos-y="0.0" pos-z="1.0"/>
            <model-instance-keyframe frame="22" pos-x="0.0" pos-y="0.0" pos-z="1.0"/>
            <model-instance-keyframe frame="21" pos-x="0.0" pos-y="0.0" pos-z="1.0"/>
        </model-instance>
    </model-instances>
    <cameras>
        <camera name="c1">
            <camera-keyframe frame-number="30" eye-x="0.0" eye-y="0.0" eye-z="1.0" target-x="0.0" target-y="0.0" target-z="0.0" up-x="0.0" up-y="0.0" up-z="0.0" field-of-view="60.0"/>
            <camera-keyframe frame-number="31" eye-x="0.0" eye-y="0.0" eye-z="1.0" target-x="0.0" target-y="0.0" target-z="0.0" up-x="0.0" up-y="0.0" up-z="0.0" field-of-view="60.0"/>
            <camera-keyframe frame-number="32" eye-x="0.0" eye-y="0.0" eye-z="1.0" target-x="0.0" target-y="0.0" target-z="0.0" up-x="0.0" up-y="0.0" up-z="0.0" field-of-view="60.0"/>
        </camera>
    </cameras>
</world3dmodel>
	*/


	XMLDocument doc;
	if (!doc.Load(pInputStream))
		return FALSE;

	if (!doc.pRootElement->strName.Equals("world3dmodel"))
		return FALSE;

	int i;
	char cbPath[1024];
	SuperString strValue;
	SuperString strKey;

	//
	// Load model definitions
	//

	int iModelDefinitions = doc.GetCount("model-definitions.model-definition");
	for (i=0; i<iModelDefinitions; i++) {
		W3D_MODEL_DEFINITION* pModelDefinition = 
			(W3D_MODEL_DEFINITION*) MemAlloc(sizeof(W3D_MODEL_DEFINITION));

		wsprintf(cbPath, "model-definitions.model-definition<%d>@name", i+1);
		doc.GetValue(&strValue, cbPath);
		strcpy(pModelDefinition->szName, strValue.GetBuffer());

		wsprintf(cbPath, "model-definitions.model-definition<%d>@type", i+1);
		doc.GetValue(&strValue, cbPath);
		strcpy(pModelDefinition->szType, strValue.GetBuffer());
		if (strValue.Equals("WAVEFRONT")) {
			wsprintf(cbPath, "model-definitions.model-definition<%d>@filename", i+1);
			doc.GetValue(&strValue, cbPath);

			// Translate relative to absolute path
			SuperFilename sf;
			char cb[512];
			sf.GetAbsoluteFilename(
				cb, strPathFilename.GetBuffer(), 
				strValue.GetBuffer());
			strcpy(pModelDefinition->szFilename, cb);

			// Load model
			WavefrontObject3dModel* p = new WavefrontObject3dModel();
			if (!p->LoadFromFile(pModelDefinition->szFilename)) {
				MemFree(pModelDefinition);
				delete p;
				return FALSE;
			}
			pModelDefinition->pObject3dModel = p;
		}
		else if (strValue.Equals("AUTODESK3DS")) {
			wsprintf(cbPath, "model-definitions.model-definition<%d>@filename", i+1);
			doc.GetValue(&strValue, cbPath);

			// Translate relative to absolute path
			SuperFilename sf;
			char cb[512];
			sf.GetAbsoluteFilename(
				cb, strPathFilename.GetBuffer(), 
				strValue.GetBuffer());
			strcpy(pModelDefinition->szFilename, cb);

			//
			// Load model
			//

			// Check if database is already loaded..
			Autodesk3dsDatabase* pDatabase = NULL;
			for (int j=0; j<vAutodesk3dsDatabases.Size(); j++) {
				Autodesk3dsDatabase* p = (Autodesk3dsDatabase*) vAutodesk3dsDatabases.ElementAt(j);
				if (p->strPathFilename.Equals(pModelDefinition->szFilename)) {
					pDatabase = p;
					break;
				}
			}
			// If not - load it..
			if (!pDatabase) {
				pDatabase = new Autodesk3dsDatabase();
				if (!pDatabase->LoadFromFile(pModelDefinition->szFilename)) {
					MemFree(pModelDefinition);
					delete pDatabase;
					return FALSE;
				}
				vAutodesk3dsDatabases.AddElement(pDatabase);
			}

			// Try to find mesh.. (if can't - no model definition..)
			A3DS_MESH* pMesh = pDatabase->GetMeshByName(pModelDefinition->szName);
			if (pMesh) {
				Autodesk3dsObject3dModel* p = new Autodesk3dsObject3dModel(pDatabase, pModelDefinition->szName);
				pModelDefinition->pObject3dModel = p;
			}
			else {
				MemFree(pModelDefinition);
				return FALSE;
			}
		}
		vModelDefinitions.AddElement(pModelDefinition);
	}

	//
	// Load frame names
	//

	int iFrameNames = doc.GetCount("frame-names.frame-name");
	for (i=0; i<iFrameNames; i++) {
		W3D_FRAME_NAME* pFrameName = 
			(W3D_FRAME_NAME*) MemAlloc(sizeof(W3D_FRAME_NAME));

		wsprintf(cbPath, "frame-names.frame-name<%d>@number", i+1);
		doc.GetValue(&strValue, cbPath);
		pFrameName->iFrameNo = strValue.IntValue();

		wsprintf(cbPath, "frame-names.frame-name<%d>@name", i+1);
		doc.GetValue(&strValue, cbPath);
		strcpy(pFrameName->szName, strValue.GetBuffer());

		vFrameNames.AddElement(pFrameName);
	}

	//
	// Load model instances
	//

	int iModelInstances = doc.GetCount("model-instances.model-instance");
	for (i=0; i<iModelInstances; i++) {
		World3dModelModelInstance *pModelInstance = 
			new World3dModelModelInstance();
		wsprintf(cbPath, "model-instances.model-instance<%d>", i+1);
		XMLElement *p = doc.GetElement(cbPath);
		if (!p)
			return FALSE;
		if (!pModelInstance->Load(p))
			return FALSE;
		vFrameObjects.AddElement(pModelInstance);
	}

	//
	// Load cameras
	//


	int iCameras = doc.GetCount("cameras.camera");
	for (i=0; i<iCameras; i++) {

		World3dModelCamera *pCamera = new World3dModelCamera();
		wsprintf(cbPath, "cameras.camera<%d>", i+1);
		XMLElement *p = doc.GetElement(cbPath);
		if (!p)
			return FALSE;
		if (!pCamera->Load(p))
			return FALSE;
		vFrameObjects.AddElement(pCamera);
	}

	return TRUE;
}


XMLElement* World3dModel::SaveModelDefinition(W3D_MODEL_DEFINITION *pModelDefinition, char* pszWorld3dModelPathFilename) {
	XMLElement* pEModelDefinition = new XMLElement("model-definition");
	SuperFilename sf;
	sf.SetPathFilename(pModelDefinition->szFilename);
	char cbRelative[512];
	sf.GetRelativeFilename(cbRelative, pszWorld3dModelPathFilename,
		pModelDefinition->szFilename);
	pEModelDefinition->AddAttribute("name", pModelDefinition->szName);
	pEModelDefinition->AddAttribute("type", pModelDefinition->szType);
	pEModelDefinition->AddAttribute("filename", cbRelative);
	return pEModelDefinition;
}



void World3dModel::SetPathFilename(char *pszPathFilename) {
	strPathFilename.Set(pszPathFilename);
}


int World3dModel::GetTotalFrames() {
	// Find max frame number
	int iTotalFrames = 0;
	int i;
	for (i=0; i<vFrameObjects.Size(); i++) {
		World3dModelFrameObject* pFrameObject = 
			(World3dModelFrameObject*) vFrameObjects.ElementAt(i);
		for (int t=0; t<pFrameObject->vKeyframes.Size(); t++) {
			World3dModelKeyframe* pKeyframe= 
				(World3dModelKeyframe*) pFrameObject->vKeyframes.ElementAt(t);
			if (pKeyframe->iFrameNo > iTotalFrames)
				iTotalFrames = pKeyframe->iFrameNo;
		}
	}
	return iTotalFrames;
}

void World3dModel::RemoveModelDefinition(char *pszModelName) {
	for (int i=0; i<vModelDefinitions.Size(); i++) {
		W3D_MODEL_DEFINITION* pModelDefinition = 
			(W3D_MODEL_DEFINITION*) vModelDefinitions.ElementAt(i);
		if (strcmp(pModelDefinition->szName, pszModelName) == 0) {
			vModelDefinitions.RemoveAt(i);
			if (pModelDefinition->pObject3dModel)
				delete pModelDefinition->pObject3dModel;
			// If this was autodesk 3ds mesh
			if (strcmp(pModelDefinition->szType, "AUTODESK3DS") == 0) {
				// and we have no more meshes in use from this database
				BOOL isUsed = FALSE;
				for (int t=0; t<vModelDefinitions.Size(); t++) {
					W3D_MODEL_DEFINITION* p = 
						(W3D_MODEL_DEFINITION*) vModelDefinitions.ElementAt(t);
					if ((strcmp(p->szType, "AUTODESK3DS") == 0) &&
						(strcmp(p->szFilename, pModelDefinition->szFilename) == 0))
					{
						isUsed = TRUE;
						break;
					}
				}
				// Then delete autodesk database
				if (!isUsed) {
					for (int j=0; j<vAutodesk3dsDatabases.Size(); j++) {
						Autodesk3dsDatabase* pAutodesk3dsDatabase = 
							(Autodesk3dsDatabase*) vAutodesk3dsDatabases.ElementAt(j);
						if (pAutodesk3dsDatabase->strPathFilename.Equals(
							pModelDefinition->szFilename))
						{
							delete pAutodesk3dsDatabase;
							vAutodesk3dsDatabases.RemoveAt(j);
							break;
						}
					}
				}
			}
			delete pModelDefinition;
			break;
		}
	}

	BOOL hasMore = TRUE;
	while(hasMore) {
		hasMore = FALSE;
		// Remove all frame objects with using this model
		for (i=0; i<vFrameObjects.Size(); i++) {
			World3dModelFrameObject* pFrameObject = 
				(World3dModelFrameObject*) vFrameObjects.ElementAt(i);
			if (pFrameObject->strType.Equals("ModelInstance")) {
				World3dModelModelInstance *pModelInstance = 
					(World3dModelModelInstance *) pFrameObject;
				if (pModelInstance->strModelName.Equals(pszModelName)) {
					RemoveFrameObject(pFrameObject->strName.GetBuffer());
					hasMore = TRUE;
					break;
				}
			}
		}
	}
}


void World3dModel::RemoveFrameObject(char* pszName) {
	for (int i=0; i<vFrameObjects.Size(); i++) {
		World3dModelFrameObject* pFrameObject = 
			(World3dModelFrameObject*) vFrameObjects.ElementAt(i);
		if (pFrameObject->strName.Equals(pszName)) {
			delete pFrameObject;
			vFrameObjects.RemoveAt(i);
			return;
		}
	}
}

void World3dModel::PopulateWorld(int iFramePosition, World3d &world3d) {
	// Scan all frame objects
	for (int i=0; i<vFrameObjects.Size(); i++) {
		World3dModelFrameObject* pFrameObject = 
			(World3dModelFrameObject*) vFrameObjects.ElementAt(i);
		// Take model instance ones
		if (pFrameObject->strType.Equals("ModelInstance")) {
			World3dModelModelInstance *pModelInstance = 
				(World3dModelModelInstance *) pFrameObject;
			// If model instance has frame of keyframe here
			if ((pModelInstance->HasFrame(iFramePosition)) ||
				(pModelInstance->HasKeyframe(iFramePosition)))
			{
				// Add it to world..
				W3D_MODEL_DEFINITION* pModelDefinition = 
					GetModelDefinition(pModelInstance->strModelName.GetBuffer());
				if (pModelDefinition) {
					ModelObject3d *pModelObject3d = new ModelObject3d();
					pModelObject3d->pObject3dModel = pModelDefinition->pObject3dModel;
					// Calc position...
					pModelInstance->ApplyToObject3d(iFramePosition, *pModelObject3d);
					world3d.AddObject3d(pModelObject3d);
				}
			}
		}
	}
}

W3D_MODEL_DEFINITION* World3dModel::GetModelDefinition(char *pszName) {
	for (int i=0; i<vModelDefinitions.Size(); i++) {
		W3D_MODEL_DEFINITION* pModelDefinition = 
			(W3D_MODEL_DEFINITION*) vModelDefinitions.ElementAt(i);
		if (strcmp(pModelDefinition->szName, pszName) == 0)
			return pModelDefinition;
	}
	return NULL;
}
