/**
 * World3dModelCamera.cpp
 * Implementation of the World3dModelCamera class.
 *
 * @author Matija Tomaskovic
 * @version 21-May-2001
 */

#include "World3dModelCamera.h"
#include <tomaskovic\util\Buffer.h>
#include <tomaskovic\util\TextWriter.h>
#include <tomaskovic\util\TextInputReader.h>
#include <tomaskovic\util\SuperString.h>
#include <tomaskovic\util\MemoryManager.h>
#include <tomaskovic\util\SuperFilename.h>
#include "WavefrontObject3dModel.h"
#include <tomaskovic\xml\XMLDocument.h>
#include <tomaskovic\xml\XMLElement.h>
#include <tomaskovic\xml\XMLAttribute.h>

World3dModelCamera::World3dModelCamera() {
	strType.Set("Camera");
}


World3dModelCamera::~World3dModelCamera() {
}


XMLElement* World3dModelCamera::Save() {
	XMLElement* pECamera = new XMLElement("camera");
	pECamera->AddAttribute("name", strName.GetBuffer());
	// Save keyframes
	for (int t=0; t<vKeyframes.Size(); t++) {
		World3dModelCameraKeyframe* pCameraKeyframe =
			(World3dModelCameraKeyframe*) vKeyframes.ElementAt(t);
		pECamera->AddChildElement(pCameraKeyframe->Save());
	}
	// Save frames
	for (t=0; t<aFrames.Size(); t++) {
		int* p = (int*) aFrames.ElementAt(t);
		XMLElement* pEFrame = pECamera->AddChildElement("camera-frame");
		pEFrame->AddAttribute("frame-number", *p);
	}

	return pECamera;
}


BOOL World3dModelCamera::Load(XMLElement* pECamera) {

	/*
        <camera name="c1">
            <camera-keyframe frame-number="30" eye-x="0.0" eye-y="0.0" eye-z="1.0" target-x="0.0" target-y="0.0" target-z="0.0" up-x="0.0" up-y="0.0" up-z="0.0" field-of-view="60.0"/>
            <camera-keyframe frame-number="31" eye-x="0.0" eye-y="0.0" eye-z="1.0" target-x="0.0" target-y="0.0" target-z="0.0" up-x="0.0" up-y="0.0" up-z="0.0" field-of-view="60.0"/>
            <camera-keyframe frame-number="32" eye-x="0.0" eye-y="0.0" eye-z="1.0" target-x="0.0" target-y="0.0" target-z="0.0" up-x="0.0" up-y="0.0" up-z="0.0" field-of-view="60.0"/>
        </camera>
	*/

	char cbPath[256];
	SuperString strValue;

	pECamera->GetValue(&strValue, "@name");
	strName.Set(strValue.GetBuffer());

	int iKeyframes = pECamera->GetCount("camera-keyframe");
	for (int t=0; t<iKeyframes; t++) {
		World3dModelCameraKeyframe* pCameraKeyframe =
			new World3dModelCameraKeyframe();

		wsprintf(cbPath, "camera-keyframe<%d>@frame-number", t+1);
		pECamera->GetValue(&strValue, cbPath);
		pCameraKeyframe->iFrameNo = strValue.IntValue();

		wsprintf(cbPath, "camera-keyframe<%d>@eye-x", t+1);
		pECamera->GetValue(&strValue, cbPath);
		pCameraKeyframe->fEyeX = strValue.FloatValue();

		wsprintf(cbPath, "camera-keyframe<%d>@eye-y", t+1);
		pECamera->GetValue(&strValue, cbPath);
		pCameraKeyframe->fEyeY = strValue.FloatValue();

		wsprintf(cbPath, "camera-keyframe<%d>@eye-z", t+1);
		pECamera->GetValue(&strValue, cbPath);
		pCameraKeyframe->fEyeZ = strValue.FloatValue();


		wsprintf(cbPath, "camera-keyframe<%d>@target-x", t+1);
		pECamera->GetValue(&strValue, cbPath);
		pCameraKeyframe->fTargetX = strValue.FloatValue();

		wsprintf(cbPath, "camera-keyframe<%d>@target-y", t+1);
		pECamera->GetValue(&strValue, cbPath);
		pCameraKeyframe->fTargetY = strValue.FloatValue();

		wsprintf(cbPath, "camera-keyframe<%d>@target-z", t+1);
		pECamera->GetValue(&strValue, cbPath);
		pCameraKeyframe->fTargetZ = strValue.FloatValue();


		wsprintf(cbPath, "camera-keyframe<%d>@up-x", t+1);
		pECamera->GetValue(&strValue, cbPath);
		pCameraKeyframe->fUpX = strValue.FloatValue();

		wsprintf(cbPath, "camera-keyframe<%d>@up-y", t+1);
		pECamera->GetValue(&strValue, cbPath);
		pCameraKeyframe->fUpY = strValue.FloatValue();

		wsprintf(cbPath, "camera-keyframe<%d>@up-z", t+1);
		pECamera->GetValue(&strValue, cbPath);
		pCameraKeyframe->fUpZ = strValue.FloatValue();


		wsprintf(cbPath, "camera-keyframe<%d>@field-of-view", t+1);
		pECamera->GetValue(&strValue, cbPath);
		pCameraKeyframe->fFOV = strValue.FloatValue();

		vKeyframes.AddElement(pCameraKeyframe);
	}

	int iFrames = pECamera->GetCount("camera-frame");
	for (t=0; t<iFrames; t++) {
		int* p = (int*) aFrames.AddElement();
		wsprintf(cbPath, "camera-frame<%d>@frame-number", t+1);
		pECamera->GetValue(&strValue, cbPath);
		*p = strValue.IntValue();
	}

	return TRUE;
}


BOOL World3dModelCamera::ApplyToCamera(int iFrameNo, Camera *pCamera) {
	if (HasFrame(iFrameNo)) {

		//
		// Calc interpolated keyframe for camera
		//

		World3dModelCameraKeyframe keyframe;
		World3dModelCameraKeyframe *pKeyframeBefore =
			(World3dModelCameraKeyframe*) GetKeyframeBefore(iFrameNo);
		World3dModelCameraKeyframe *pKeyframeAfter =
			(World3dModelCameraKeyframe*) GetKeyframeAfter(iFrameNo);

		if ((!pKeyframeBefore) || (!pKeyframeAfter))
			// ERROR: this shouldn't happen - no keyframe before frame
			return FALSE;

		float fPos = iFrameNo;
		float fBefore = pKeyframeBefore->iFrameNo;
		float fAfter = pKeyframeAfter->iFrameNo;
		float fAmount = (fPos - fBefore) / (fAfter - fBefore);

		keyframe.fEyeX = pKeyframeBefore->fEyeX +
			(pKeyframeAfter->fEyeX - pKeyframeBefore->fEyeX) * fAmount;
		keyframe.fEyeY = pKeyframeBefore->fEyeY +
			(pKeyframeAfter->fEyeY - pKeyframeBefore->fEyeY) * fAmount;
		keyframe.fEyeZ = pKeyframeBefore->fEyeZ +
			(pKeyframeAfter->fEyeZ - pKeyframeBefore->fEyeZ) * fAmount;

		keyframe.fTargetX = pKeyframeBefore->fTargetX +
			(pKeyframeAfter->fTargetX - pKeyframeBefore->fTargetX) * fAmount;
		keyframe.fTargetY = pKeyframeBefore->fTargetY +
			(pKeyframeAfter->fTargetY - pKeyframeBefore->fTargetY) * fAmount;
		keyframe.fTargetZ = pKeyframeBefore->fTargetZ +
			(pKeyframeAfter->fTargetZ - pKeyframeBefore->fTargetZ) * fAmount;

		keyframe.fUpX = pKeyframeBefore->fUpX +
			(pKeyframeAfter->fUpX - pKeyframeBefore->fUpX) * fAmount;
		keyframe.fUpY = pKeyframeBefore->fUpY +
			(pKeyframeAfter->fUpY - pKeyframeBefore->fUpY) * fAmount;
		keyframe.fUpZ = pKeyframeBefore->fUpZ +
			(pKeyframeAfter->fUpZ - pKeyframeBefore->fUpZ) * fAmount;

		keyframe.fFOV = pKeyframeBefore->fFOV +
			(pKeyframeAfter->fFOV - pKeyframeBefore->fFOV) * fAmount;

		keyframe.ApplyToCamera(pCamera);
		return TRUE;
	}
	else if (HasKeyframe(iFrameNo)) {
		World3dModelCameraKeyframe *pKeyframe =
			(World3dModelCameraKeyframe*) GetKeyframe(iFrameNo);
		pKeyframe->ApplyToCamera(pCamera);
		return TRUE;
	}

	return FALSE;
}

void World3dModelCamera::SaveCameraStateAsKeyframe(int iFrameNo, Camera *pCamera) {
	if (HasKeyframe(iFrameNo)) {
		World3dModelCameraKeyframe *pKeyframe =
			(World3dModelCameraKeyframe*) GetKeyframe(iFrameNo);
		pKeyframe->fEyeX = pCamera->fEyeX;
		pKeyframe->fEyeY = pCamera->fEyeY;
		pKeyframe->fEyeZ = pCamera->fEyeZ;
		pKeyframe->fTargetX = pCamera->fTargetX;
		pKeyframe->fTargetY = pCamera->fTargetY;
		pKeyframe->fTargetZ = pCamera->fTargetZ;
		pKeyframe->fUpX = pCamera->fUpX;
		pKeyframe->fUpY = pCamera->fUpY;
		pKeyframe->fUpZ = pCamera->fUpZ;
		pKeyframe->fFOV = pCamera->fFOV;
	}
	else {
		World3dModelCameraKeyframe *pKeyframe =
			new World3dModelCameraKeyframe();
		pKeyframe->iFrameNo = iFrameNo;
		pKeyframe->fEyeX = pCamera->fEyeX;
		pKeyframe->fEyeY = pCamera->fEyeY;
		pKeyframe->fEyeZ = pCamera->fEyeZ;
		pKeyframe->fTargetX = pCamera->fTargetX;
		pKeyframe->fTargetY = pCamera->fTargetY;
		pKeyframe->fTargetZ = pCamera->fTargetZ;
		pKeyframe->fUpX = pCamera->fUpX;
		pKeyframe->fUpY = pCamera->fUpY;
		pKeyframe->fUpZ = pCamera->fUpZ;
		pKeyframe->fFOV = pCamera->fFOV;
		vKeyframes.AddElement(pKeyframe);
	}

}
