/****************************************************************************
 * SuperLog.cpp: implementation of the SuperLog class.
 * Check SuperLog.h for details...
 ***************************************************************************/

#include "SuperLog.h"
#include "Buffer.h"
#include "Vector.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Vector g_vLogs;

SuperLog::SuperLog(char *pszName, char *pszPathFilename, bool bDebugOnly)
{
	strcpy(m_szName, pszName);
	strcpy(m_szPathFilename, pszPathFilename);
	m_bDebugOnly = bDebugOnly;
	g_vLogs.AddElement(this);
}


SuperLog::~SuperLog()
{
	logLnWithString(m_szName, "*** End of logging for '%s' ***", m_szName);
	g_vLogs.RemoveElement(this);
}



/**
 * Prepares log
 * @param pszName			- symbolic name
 * @param pszPathFilename	- actual filename where to save log
 */
void SuperLog::PrepareLog(char *pszName, char *pszPathFilename)
{
	PrepareLog(pszName, pszPathFilename, true);
}

/**
 * Prepares log
 * @param pszName			- symbolic name
 * @param pszPathFilename	- actual filename where to save log
 * @param bDebugOnly		- if true, log will be active only when _DEBUG_
 */
void SuperLog::PrepareLog(char *pszName, char *pszPathFilename, bool bDebugOnly)
{
	// try to find existing one..
	for (int i=0; i<g_vLogs.Size(); i++) {
		SuperLog* p = (SuperLog*) g_vLogs.ElementAt(i);
		if (strcmp(p->m_szName, pszName) == 0)
			return;
	}

	SuperLog *pLog = new SuperLog(pszName, pszPathFilename, bDebugOnly);
	g_vLogs.AddElement(pLog);
	pLog->logLnWithString(pszName, "*** Started logging %s ***", pszName);
}


void SuperLog::logLn(char *pszLogName, char *pszText)
{
	// find log..
	for (int i=0; i<g_vLogs.Size(); i++) {
		SuperLog* p = (SuperLog*) g_vLogs.ElementAt(i);
		if (strcmp(pszLogName, p->m_szName) == 0) {
			char *b = new char[strlen(pszText) + 1 + strlen(p->m_szName) + 1];
			wsprintf(b, "(%s) %s" , p->m_szName, pszText);
			p->logLn(b);
			return;
		}
	}
}


void SuperLog::logLnWithBlockString(char* pszLogName, char *pszFormat, char* pszText, unsigned long ulStrLen)
{
	// Copy into new buffer until EOT or ulStrLen, and finish with EOT

	char* pBuf = new char[ulStrLen+1];
	unsigned long i = 0;

	while(i<ulStrLen)
	{
		if (*(pszText+i) == '\0')
			break;

		if (i+1<ulStrLen) {
			if ((*(pszText+i) == ' ') && (*(pszText+i+1) == ' '))
				break;
		}
		*(pBuf+i) = *(pszText+i);
		i++;
	}

	*(pBuf+i) = '\0';


	char* p = new char[strlen(pBuf) + strlen(pszFormat)*2];
	wsprintf(p, pszFormat, pBuf);
	logLn(pszLogName, p);

	delete p;
	delete pBuf;
}


void SuperLog::logLnWithInt(char *pszLogName, char *pszFormat, int iValue)
{
	char *p = new char[strlen(pszFormat) + 10 + 2048];
	wsprintf(p, pszFormat, iValue);
	logLn(pszLogName, p);
	delete p;
}

void SuperLog::logLnWithIntInt(char *pszLogName, char *pszFormat, int iVal1, int iVal2)
{
	char *p = new char[strlen(pszFormat) + 10 + 2048];
	wsprintf(p, pszFormat, iVal1, iVal2);
	logLn(pszLogName, p);
	delete p;
}

void SuperLog::logLnWithString(char *pszLogName, char *pszFormat, char *pszValue)
{
	char *p = new char[strlen(pszFormat) + strlen(pszValue) * 3 + 2048];
	wsprintf(p, pszFormat, pszValue);
	logLn(pszLogName, p);
	delete p;
}

void SuperLog::logLnWithHexBlock(char *pszLogName, char *pszFormat, BYTE *pData, int iDataLength)
{
	Buffer buffer;
	buffer.EnsureSize(iDataLength*3 + strlen(pszFormat) + 500);	// every number = '30_' + terminating '\0'
	char* p = (char*) buffer.GetBuffer();
	char letters[] = "0123456789ABCDEF";
	for (int i=0; i<iDataLength; i++) {
		int num = *(pData + i);
		int high = (num & 0xf0) / 16;
		int low = num & 0x0f;
		*p++ = letters[high];
		*p++ = letters[low];
		*p++ = ' ';
	}
	*p++ = '\0';

	logLnWithString(pszLogName, pszFormat, (char*) buffer.GetBuffer());
}



/****************************************************************************
 * void SuperLog::ClearLogList()
 ***************************************************************************/
void SuperLog::ClearLogList()
{
	g_vLogs.Clear();
}

/****************************************************************************
 * void SuperLog::logLn(char *pszText)
 ***************************************************************************/
void SuperLog::logLn(char *pszText)
{
	// If not debug (but release) version - obey bDebugOnly flag...
#ifndef _DEBUG
	if (!m_bDebugOnly) {
#endif _DEBUG

	FILE *pFile = fopen(m_szPathFilename, "a");
	if (pFile) {
		SYSTEMTIME st;
		GetLocalTime(&st);
		char buf[2048];
		wsprintf(buf, "%04d-%02d-%02d|%02d:%02d:%02d.%04d> %s\n", st.wYear, st.wMonth, st.wDay,
				st.wHour, st.wMinute, st.wSecond, st.wMilliseconds, pszText);
		fwrite(buf, strlen(buf), 1, pFile);
		fclose(pFile);
	}

#ifndef _DEBUG
	}
#endif _DEBUG

}

