/**
 * XMLDocument.cpp
 *
 * @author Matija Tomaskovic
 * @version 11-Jul-2001
 */

#include "XMLDocument.h"
#include "XMLElement.h"
#include "XMLAttribute.h"
#include <tomaskovic\util\TextInputReader.h>
#include <tomaskovic\util\TextWriter.h>


XMLDocument::XMLDocument() {
	pRootElement = NULL;
}


XMLDocument::~XMLDocument() {
	if (pRootElement)
		delete pRootElement;
}



BOOL XMLDocument::Load(InputStream *pInputStream) {
	/*
		<?xml version="1.0" encoding="ISO-8859-2"?>
		<trgovanje>
		   <korisnik ime="Karlo" prezime="Magdic" adresa="prava adresa"/>
		   <bez-dionice/>
		   <sa-dionicom>
			  <dionica id="121" naziv="Varazdinska Banka" vrijednost="211kn"/>
		   </sa-dionicom>
		</trgovanje>
	*/
    TextInputReader tir(pInputStream);
	if (!tir.ReadNextChar('<')) return Error(&tir);
	if (!tir.ReadNextChar('?')) return Error(&tir);
	if (!tir.ReadNextWord("xml")) return Error(&tir);
	tir.SkipSpaces();
	if (!tir.ReadWordUntilDelimiter('=')) return Error(&tir);
	if (!tir.strLastWord.Equals("version")) return Error("'version' expected");
	if (!tir.ReadNextChar('=')) return Error(&tir);
	if (!tir.ReadNextString('"')) return Error(&tir);
	// we have version...

	tir.SkipSpaces();
	if (!tir.ReadWordUntilDelimiter('=')) return Error(&tir);
	if (!tir.strLastWord.Equals("encoding")) return Error("'version' expected");
	if (!tir.ReadNextChar('=')) return Error(&tir);
	if (!tir.ReadNextString('"')) return Error(&tir);
	// we have encoding...

	if (!tir.ReadNextChar('?')) return Error(&tir);
	if (!tir.ReadNextChar('>')) return Error(&tir);
	tir.SkipSpacesAndEOL();

	//
	// Read root element
	//

	tir.SkipSpacesAndEOL();
	if (!tir.ReadNextChar('<')) return Error(&tir);
	tir.PushBack('<');
	XMLElement *pXMLElement = LoadElement(&tir);
	if (!pXMLElement) 
		return FALSE;
	pRootElement = pXMLElement;

	return TRUE;
}

BOOL XMLDocument::Error(TextInputReader* pTextInputReader) {
	return Error(pTextInputReader->strLastError.GetBuffer());
}

BOOL XMLDocument::Error(char* pszDescription) {
	MessageBox(NULL, pszDescription, "ERROR", MB_OK);
	return FALSE;
}


XMLElement* XMLDocument::LoadElement(TextInputReader *pTextInputReader) {
	pTextInputReader->SkipSpacesAndEOL();
	if (!pTextInputReader->ReadNextChar('<')) {
		Error(pTextInputReader);
		return NULL;
	}

	//
	// Read element name (until space, '/' or '>')
	//

	SuperString strName;
	int iChar;
	while(TRUE) {
		iChar = pTextInputReader->ReadChar();
		if (iChar == -1) {
			Error("Element name expected");
			return NULL;
		}
		if ((iChar == '/') || (iChar == ' ') || (iChar == '>'))
			break;
		strName.Append((char) iChar);
	}

	XMLElement *pElement = new XMLElement(strName.GetBuffer());

	//
	// Read attributes
	//

	//       | we are here  
	// <world name="big"... /> or
	//                      | should come here
	// <world name="big"... >
	//                      | or here
	// <world/>
	//       | or here
	// <world>
	//       | or here


	while(TRUE) {
		// Attribute follows?
		if (iChar == ' ') {
			pTextInputReader->SkipSpaces();
			SuperString strAttributeName;
			while(TRUE) {
				iChar = pTextInputReader->ReadChar();
				if (iChar == -1) {
					Error("Attribute name or end of element expected");
					delete pElement;
					return NULL;
				}
				if ((iChar == '/') || 
					(iChar == '>') || 
					(iChar == '='))
					break;
				strAttributeName.Append((char) iChar);
			}

			if (iChar == '=') {
				// attribute..
				if (!pTextInputReader->ReadNextString('"')) {
					Error(pTextInputReader);
					delete pElement;
					return NULL;
				}
				pElement->AddAttribute(strAttributeName.GetBuffer(), pTextInputReader->strLastWord.GetBuffer());
			}
			else {
				Error("Attribute value expected");
				delete pElement;
				return NULL;
			}

			iChar = pTextInputReader->ReadChar();
			if (iChar == -1) {
				Error("EOF not expected while in element definition");
				delete pElement;
				return NULL;
			}
		}
		else if (iChar == '>') {
			//
			// data, child elements and end-element will follow..
			//
			break;
		}
		else if (iChar == '/') {
			// "../>" end of element...
			if (!pTextInputReader->ReadNextChar('>')) {
				Error(pTextInputReader);
				delete pElement;
				return NULL;
			}
			else {
				return pElement;
			}
		}
	}

	//
	// data, child elements and end-element will follow..
	//

	// <world name="big"... >
	//                       | we are here
	// </world>

	while(TRUE) {
		pTextInputReader->SkipSpacesAndEOL();
		int iChar = pTextInputReader->ReadChar();
		if (iChar == '<') {
			iChar = pTextInputReader->ReadChar();
			if (iChar == '/') {
				//
				// End of element
				//
				if (!pTextInputReader->ReadWordUntilDelimiter('>')) {
					Error(pTextInputReader);
					delete pElement;
					return NULL;
				}
				// check if same as element name
				if (!pTextInputReader->strLastWord.Equals(pElement->strName.GetBuffer())) {
					Error("Bad element name in ending tag");
					delete pElement;
					return NULL;
				}
				if (!pTextInputReader->ReadNextChar('>')) {
					Error(pTextInputReader);
					delete pElement;
					return NULL;
				}
				return pElement;
			}
			else {
				//
				// Child element...
				//
				pTextInputReader->PushBack(iChar);
				pTextInputReader->PushBack('<');
				XMLElement* pChild = LoadElement(pTextInputReader);
				pElement->AddChildElement(pChild);
				continue;
			}
		}
		else {
			SuperString strData;
			while(TRUE) {
				// Data follows - read until <
				int iChar = pTextInputReader->ReadChar();
				if (iChar == -1) {
					// EOF not expected here
					delete pElement;
					return NULL;
				}
				if (iChar == '<') {
					pTextInputReader->PushBack(iChar);
					break;
				}
				strData.Append((char) iChar);
			}
			pElement->strValue.Set(strData.GetBuffer());
		}
	}
}


BOOL XMLDocument::GetValue(SuperString *pstrValueDest, char *pszPath) {
	return pRootElement->GetValue(pstrValueDest, pszPath);
}


int XMLDocument::GetCount(char *pszPath) {
	return pRootElement->GetCount(pszPath);
}


void XMLDocument::Save(OutputStream *pOutputStream) {
	TextWriter tw(pOutputStream);
	int iTabs = 0;
	tw.WriteLine("<?xml version=\"1.0\" encoding=\"ISO-8859-2\"?>");
	SaveElement(pOutputStream, 0, pRootElement);
}

void XMLDocument::SaveElement(OutputStream *pOutputStream, int iTabs, XMLElement *pElement) {
	TextWriter tw(pOutputStream);

	// Write: "<elementname"
	for (int t=0; t<iTabs; t++)
		tw.Write("    ");

	tw.Write("<");
	tw.Write(pElement->strName.GetBuffer());

	// Write attributes: ... at="value"
	for (int i=0; i<pElement->vAttributes.Size(); i++) {
		XMLAttribute *p = (XMLAttribute*) pElement->vAttributes.ElementAt(i);
		tw.Write(" ");
		tw.Write(p->strName.GetBuffer());
		tw.Write("=\"");
		tw.Write(p->strValue.GetBuffer());
		tw.Write("\"");
	}

	// If has no child elements and no data: "/>" and new line
	if (pElement->vChildElements.Size() == 0) {
		if (pElement->strValue.GetLength() == 0) {
			tw.Write("/>");
			tw.WriteEOL();
			return;
		}
		else {
			tw.Write(">");
			tw.Write(pElement->strValue.GetBuffer());
			tw.Write("</");
			tw.Write(pElement->strName.GetBuffer());
			tw.Write(">");
			tw.WriteEOL();
		}
	}
	else {
		tw.Write(">");
		tw.WriteEOL();

		for (int i=0; i<pElement->vChildElements.Size(); i++) {
			XMLElement* p = (XMLElement*) pElement->vChildElements.ElementAt(i);
			SaveElement(pOutputStream, iTabs+1, p);
		}

		for (int t=0; t<iTabs; t++)
			tw.Write("    ");

		if (pElement->strValue.GetLength() == 0)
			tw.Write(pElement->strValue.GetBuffer());

		tw.Write("</");
		tw.Write(pElement->strName.GetBuffer());
		tw.Write(">");
		tw.WriteEOL();
	}
}

XMLElement* XMLDocument::GetElement(char *pszPath) {
	return pRootElement->GetElement(pszPath);
}
